/* platinum-builder/Pt-merge.c
 *  (c)2008, Laurence Withers.
 *
 *  Copies a source file to a destination, but only if different. Handles various types of source
 *  file (directory, symlink).
 */

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>



/* enum file_mode
 *  The various types of file or file-like-object that we deal with.
 */
enum file_mode {
    file_mode_dir,
    file_mode_link,
    file_mode_file
};



/* stat_to_file_mode
 *  Extracts the file mode (enum file_mode) from a struct stat, exiting on error.
 */
void stat_to_file_mode(struct stat* st, enum file_mode* m)
{
    if(S_ISREG(st->st_mode)) {
        *m = file_mode_file;
        return;
    }

    if(S_ISLNK(st->st_mode)) {
        *m = file_mode_link;
        return;
    }

    if(S_ISDIR(st->st_mode)) {
        *m = file_mode_dir;
        return;
    }

    fprintf(stderr, "stat_to_file_mode: Unhandled mode 0%o\n", st->st_mode);
    exit(3);
}



/* my_*()
 *  Error-checking convenience functions which exit on error.
 *   my_readlink: like readlink(2) but null-terminates its buffer
 *   my_unlink: as unlink(2)
 *   my_symlink: as symlink(2)
 *   my_chmod: as chmod(2)
 *   my_mkdir: as mkdir(2) with mode 0777
 *   my_open_rd: like open(2) with O_RDONLY
 *   my_creat: like creat(2) with mode 0666
 */
void my_readlink(const char* src, char* buf, size_t buflen)
{
    ssize_t rd;

    rd = readlink(src, buf, buflen - 1);
    if(rd == -1) {
        fprintf(stderr, "readlink(\"%s\") failed: %m\n", src);
        exit(2);
    }
    buf[rd] = 0;
}

void my_unlink(const char* path)
{
    if(unlink(path)) {
        fprintf(stderr, "unlink(\"%s\") failed: %m\n", path);
        exit(2);
    }
}

void my_symlink(const char* dest, const char* link)
{
    if(symlink(dest, link)) {
        fprintf(stderr, "symlink(\"%s\", \"%s\") failed: %m\n", dest, link);
        exit(2);
    }
}

void my_rmdir(const char* path)
{
    if(rmdir(path) == -1) {
        fprintf(stderr, "Unable to remove directory \"%s\": %m\n", path);
        exit(2);
    }
}

void my_chmod(const char* path, int mode)
{
    if(chmod(path, mode)) {
        fprintf(stderr, "chmod(\"%s\", 0%o) failed: %m\n", path, mode);
        exit(2);
    }
}

void my_mkdir(const char* path)
{
    if(mkdir(path, 0777)) {
        fprintf(stderr, "mkdir(\"%s\") failed: %m\n", path);
        exit(2);
    }
}

int my_open_rd(const char* path)
{
    int fd;
    fd = open(path, O_RDONLY);
    if(fd == -1) {
        fprintf(stderr, "open(\"%s\") failed: %m\n", path);
        exit(2);
    }
    return fd;
}

int my_creat(const char* path)
{
    int fd;
    fd = creat(path, 0666);
    if(fd == -1) {
        fprintf(stderr, "creat(\"%s\") failed: %m\n", path);
        exit(2);
    }
    return fd;
}



/* do_copy()
 *  Copy `src' to `dest'. `updating' and `print_name' are used for display purposes; `updating' is
 *  set to 0 if `dest' is being created or non-0 if it is being updated.
 */
void do_copy(const char* dest, const char* src, struct stat* src_stat, int updating,
    const char* print_name)
{
    enum file_mode src_mode;
    char linkname[PATH_MAX];
    char buf[16384];
    ssize_t rd;
    int ifd, ofd;
    const char* print_action;

    stat_to_file_mode(src_stat, &src_mode);
    print_action = updating ? "Updated" : "Created";

    switch(src_mode) {
    case file_mode_dir:
        my_mkdir(dest);
        my_chmod(dest, src_stat->st_mode & 07777);
        printf(" * %s directory %s\n", print_action, print_name);
        return;

    case file_mode_link:
        my_readlink(src, linkname, sizeof(linkname));
        my_symlink(linkname, dest);
        printf(" * %s symlink %s -> %s\n", print_action, print_name, linkname);
        return;

    case file_mode_file:
        ifd = my_open_rd(src);
        ofd = my_creat(dest);
        while(1) {
            rd = read(ifd, buf, sizeof(buf));
            if(rd == -1) {
                fprintf(stderr, "read from \"%s\" failed: %m\n", src);
                exit(2);
            }
            if(!rd) break;

            if(write(ofd, buf, rd) != rd) {
                fprintf(stderr, "write to \"%s\" failed: %m\n", dest);
                exit(2);
            }
        }
        close(ifd);
        close(ofd);
        my_chmod(dest, src_stat->st_mode & 07777);

        printf(" * %s file %s (%lu bytes)%s%s\n", print_action, print_name,
            (unsigned long)(src_stat->st_size),
            (src_stat->st_mode & 04000) ? " [SETUID]" : "",
            (src_stat->st_mode & 02000) ? " [SETGID]" : "");
        return;
    }
}



/* file_cmp()
 *  Compares `a' and `b', returning 0 if they are identical or non-0 if not.
 */
int file_cmp(const char* a, const char* b)
{
    int afd, bfd;
    char abuf[16384], bbuf[16384];
    ssize_t rd, rd2;
    int diff = 0;

    afd = my_open_rd(a);
    bfd = my_open_rd(b);

    while(1) {
        rd = read(afd, abuf, sizeof(abuf));
        if(rd == -1) {
            fprintf(stderr, "Unable to read from \"%s\": %m\n", a);
            exit(2);
        }
        if(!rd) break;

        rd2 = read(bfd, bbuf, sizeof(bbuf));
        if(rd2 == -1) {
            fprintf(stderr, "Unable to read from \"%s\": %m\n", b);
            exit(2);
        }
        if(rd != rd2) {
            fprintf(stderr, "WARNING: different read sizes for \"%s\" and \"%s\" (shouldn't happen).", a, b);
            diff = 1;
            break;
        }

        if(memcmp(abuf, bbuf, rd)) {
            diff = 1;
            break;
        }
    }

    close(afd);
    close(bfd);
    return diff;
}



/* main()
 *  Determines action based upon type of `src' and `dest', passed on the commandline.
 */
int main(int argc, char* argv[])
{
    const char* src, * dest, * print_name;
    struct stat dest_st, src_st;
    enum file_mode dest_mode, src_mode;
    char linkname1[PATH_MAX], linkname2[PATH_MAX];

    if(argc != 4) {
        fputs("Expecting three arguments: destination and src filenames, presentation name.\n", stderr);
        return 1;
    }
    dest = argv[1];
    src = argv[2];
    print_name = argv[3];

    if(lstat(src, &src_st) == -1) {
        fprintf(stderr, "lstat(\"%s\") failed: %m\n", src);
        return 2;
    }

    /* if the dest file doesn't exist, just copy it */
    if(lstat(dest, &dest_st) == -1) {
        if(errno == ENOENT) {
            do_copy(dest, src, &src_st, 0, print_name);
            return 0;
        }
        fprintf(stderr, "lstat(\"%s\") failed: %m\n", dest);
        return 2;
    }

    /* compare src and dest types */
    stat_to_file_mode(&dest_st, &dest_mode);
    stat_to_file_mode(&src_st, &src_mode);

    if(src_mode == dest_mode) {
        switch(src_mode) {
        case file_mode_dir:
            break; /* destination directory already exists */

        case file_mode_link:
            my_readlink(src, linkname1, sizeof(linkname1));
            my_readlink(dest, linkname2, sizeof(linkname2));

            if(strcmp(linkname1, linkname2)) {
                my_unlink(dest);
                do_copy(dest, src, &src_st, 1, print_name);
            }
            break;

        case file_mode_file:
            if(src_st.st_size != dest_st.st_size || file_cmp(src, dest)) {
                my_unlink(dest);
                do_copy(dest, src, &src_st, 1, print_name);
            }

            if(src_st.st_mode != dest_st.st_mode) {
                my_chmod(dest, src_st.st_mode & 07777);
                printf(" * Updated permissions on file %s %s%s\n", print_name,
                    (src_st.st_mode & 04000) ? " [SETUID]" : "",
                    (src_st.st_mode & 02000) ? " [SETGID]" : "");
            }

            break;
        }

    } else {
        if(dest_mode == file_mode_dir) my_rmdir(dest);
        else my_unlink(dest);

        do_copy(dest, src, &src_st, 1, print_name);
    }

    return 0;
}



/* vim: ts=4:sw=4:expandtab
*/
