/* libiso8601/src/isodate/000_TopSource.c
 *
 *  (c)2007, Laurence Withers, <l@lwithers.me.uk>.
 *  Released under the GNU GPLv2. See file COPYING or
 *  http://www.gnu.org/copyleft/gpl.html for details.
*/

/* Below are all the includes used throughout the application. */
#include <stdio.h>
#include <getopt.h>
#include <stdlib.h>
#include "iso8601.h"



void usage(void)
{
    fputs("Usage:\n\n"
        "    isodate [options] [date ...]\n\n"
        "Valid options:\n"
        " -h, --help                Display this screen.\n"
        " -e, --extended            Use extended display format.\n"
        " -y, --year                Only display year.\n"
        " -n, --month               Display year, month.\n"
        " -d, --day                 Display year, month, day [default].\n"
        " -o, --ordinal             Display year, ordinal day.\n"
        " -w, --week                Display ISO year, ISO week.\n"
        " -W, --week-day            Display ISO year, ISO week, ISO weekday.\n"
        " -r, --hour                Display hour.\n"
        " -R, --hour-fraction       Display hour and fraction of hour.\n"
        " -m, --minute              Display hour, minute.\n"
        " -M, --minute-fraction     Display hour, minute and fraction of\n"
        "                               minute.\n"
        " -s, --second              Display hour, minute, second.\n"
        " -S, --second-fraction     Display hour, minute, second and fraction\n"
        "                               of second [default].\n"
        " -U, --utc-offset <n>      Display time as if timezone offset was\n"
        "                               <n> seconds.\n"
        " -Z, --utc                 Display time in UTC.\n"
        "\n"
        "If specified, one or more ISO8601 dates will be parsed and displayed\n"
        "in the format given. Otherwise, the system clock will be queried and\n"
        "its time displayed.\n"
    "\n", stdout);
}



struct option options[] = {
    { "help", no_argument, 0, 'h' },
    { "extended", no_argument, 0, 'e' },
    { "year", no_argument, 0, 'y' },
    { "month", no_argument, 0, 'n' },
    { "day", no_argument, 0, 'd' },
    { "ordinal", no_argument, 0, 'o' },
    { "week", no_argument, 0, 'w' },
    { "week-day", no_argument, 0, 'W' },
    { "hour", no_argument, 0, 'r' },
    { "hour-fraction", no_argument, 0, 'R' },
    { "minute", no_argument, 0, 'm' },
    { "minute-fraction", no_argument, 0, 'M' },
    { "second", no_argument, 0, 's' },
    { "second-fraction", no_argument, 0, 'S' },
    { "utc-offset", required_argument, 0, 'U' },
    { "utc", no_argument, 0, 'Z' },
    { 0, 0, 0, 0 }
};



int main(int argc, char* argv[])
{
    struct iso8601_details det = {
        .date_prec = iso8601_prec_day,
        .time_prec = iso8601_prec_secfrac,
        .extended = 0,
        .tz_sec = 0
    };
    int override_tz = 0, ret = 0;
    char* endp, datestr[100];
    struct iso8601_date date;
    struct iso8601_details det2;

    while(1) {
        switch(getopt_long(argc, argv, "heyndowWrRmMsSU:Z", options, 0)) {
        case -1: goto opts_done;
        case '?': return 1;
        case 'h': usage(); return 0;
        case 'e': det.extended = 1; break;
        case 'y': det.date_prec = iso8601_prec_year; break;
        case 'n': det.date_prec = iso8601_prec_month; break;
        case 'd': det.date_prec = iso8601_prec_day; break;
        case 'o': det.date_prec = iso8601_prec_ord; break;
        case 'w': det.date_prec = iso8601_prec_week; break;
        case 'W': det.date_prec = iso8601_prec_wday; break;
        case 'r': det.time_prec = iso8601_prec_hour; break;
        case 'R': det.time_prec = iso8601_prec_hourfrac; break;
        case 'm': det.time_prec = iso8601_prec_min; break;
        case 'M': det.time_prec = iso8601_prec_minfrac; break;
        case 's': det.time_prec = iso8601_prec_sec; break;
        case 'S': det.time_prec = iso8601_prec_secfrac; break;
        case 'Z': det.tz_sec = 0; override_tz = 1; break;
        case 'U':
            endp = 0;
            det.tz_sec = strtol(optarg, &endp, 0);
            if(!endp || *endp || det.tz_sec <= -86400 || det.tz_sec >= 86400) {
                fputs("Invalid UTC offset. Must be an integral number of seconds\n"
                    "between -86400 and 86400, exclusive.\n", stderr);
                return 1;
            }

            override_tz = 1;
        }
    }

  opts_done:
    if(optind == argc) {
        iso8601_now(&date, &det2);
        if(!override_tz) det.tz_sec = det2.tz_sec;

        iso8601_print(datestr, sizeof(datestr), &date, &det);
        fputs(datestr, stdout);
        putc('\n', stdout);
    } else {
        for(; optind != argc; ++optind) {
            if(iso8601_parse(argv[optind], &date, 0, &det2)) {
                fputs("Couldn't parse date.\n", stderr);
                ret = 1;
                continue;
            }
            if(!override_tz) det.tz_sec = det2.tz_sec;

            iso8601_print(datestr, sizeof(datestr), &date, &det);
            fputs(datestr, stdout);
            putc('\n', stdout);
        }
    }

    return ret;
}



/* options for text editors
kate: replace-trailing-space-save true; space-indent true; tab-width 4;
vim: expandtab:ts=4:sw=4
*/
