/* libiso8601/src/libiso8601/100_leap.c
 *
 *  (c)2006-2010, Laurence Withers, <l@lwithers.me.uk>.
 *  Released under the GNU GPLv3. See file COPYING or
 *  http://www.gnu.org/copyleft/gpl.html for details.
*/



/* leap_second_days_table[]
 *  This is a sorted array of days (in libiso8601 format) on which a positive
 *  leap second occurred. This table is built in to the library, but may be
 *  overridden at runtime by changing what ‘leap_second_days’ points to. There
 *  should be no other references to this table, therefore.
 */
static int
leap_second_days_table[] = {
    720439, /* 1972-06-30 */
    720623, /* 1972-12-31 */
    720988, /* 1973-12-31 */
    721353, /* 1974-12-31 */
    721718, /* 1975-12-31 */
    722084, /* 1976-12-31 */
    722449, /* 1977-12-31 */
    722814, /* 1978-12-31 */
    723179, /* 1979-12-31 */
    723726, /* 1981-06-30 */
    724091, /* 1982-06-30 */
    724456, /* 1983-06-30 */
    725187, /* 1985-06-30 */
    726101, /* 1987-12-31 */
    726832, /* 1989-12-31 */
    727197, /* 1990-12-31 */
    727744, /* 1992-06-30 */
    728109, /* 1993-06-30 */
    728474, /* 1994-06-30 */
    729023, /* 1995-12-31 */
    729570, /* 1997-06-30 */
    730119, /* 1998-12-31 */
    732676, /* 2005-12-31 */
    733772, /* 2008-12-31 */
};



/* leap_second_days[], leap_second_days_num
 *  Pointer to an array (and number of elements in the array) representing the
 *  table of positive leap seconds. The array is sorted and contains libiso8601
 *  day numbers of days with positive leap seconds.
 */
static int*
leap_second_days = leap_second_days_table;
static int
leap_second_days_num = sizeof(leap_second_days_table) / sizeof(int);



/* iso8601_seconds_leap()
 *  Returns the number of seconds that elapse on a given date, which requires us
 *  to search the table of leap seconds to see if we need to return a special
 *  case.
 */
int
iso8601_seconds_leap(const struct iso8601_date* date)
{
    int i;
    for(i = 0; i < leap_second_days_num; ++i) {
        if(leap_second_days[i] == date->day) return 86401;
    }
    return 86400;
}



/* _leap_elapsed_day()
 *  Returns the number of leap seconds that have elapsed between ‘sday’ (start
 *  day) and ‘eday’ (end day), both in libiso8601 format.
 */
static int
_leap_elapsed_day(int sday, int eday)
{
    int spos, epos;

    for(spos = 0; spos < leap_second_days_num; ++spos) {
        if(sday <= leap_second_days[spos]) break;
    }
    for(epos = 0; epos < leap_second_days_num; ++epos) {
        if(eday <= leap_second_days[epos]) break;
    }

    return epos - spos;
}



/* iso8601_leap_elapsed()
 *  Wrapper around _leap_elapsed_day().
 */
int
iso8601_leap_elapsed(const struct iso8601_date* start,
    const struct iso8601_date* end)
{
    return _leap_elapsed_day(start->day, end->day);
}



/* leap_table_free_old
 *  If set, then when we update the leap table, we should free the old array.
 *  Initially clear so that we don't free our static built-in table.
 */
static int leap_table_free_old = 0;



/* iso8601_leap_table_set()
 *  Switch to using a new table of leap seconds, possibly freeing the old one.
 */
void
iso8601_leap_table_set(int* new_table, int new_size)
{
    if(leap_table_free_old) free(leap_second_days);
    leap_table_free_old = 0;

    leap_second_days = new_table;
    leap_second_days_num = new_size;
}



/* leap_table_signature
 *  The first 8 bytes of a leap second table must match this string, otherwise
 *  the file is taken not to be of the correct format.
 */
static const char*
leap_table_signature = "/O9PdPZI";



/* iso8601_leap_table_load()
 *  Loads a new table of leap seconds from disk. Ensures the signature of the
 *  given file matches ‘leap_table_signature’, and does some basic sanity
 *  checking (file in sorted order etc.).
 */
int
iso8601_leap_table_load(const char* fname)
{
    struct stat st;
    int fd, saved_errno, i, new_size, * days = 0;
    char buf[12];

    if(!fname) fname = DEFAULT_LEAP_TABLE;

    if(stat(fname, &st)) return -1;
    if(st.st_size < 12) {
        errno = EINVAL;
        return -1;
    }

    fd = open(fname, O_RDONLY);
    if(fd == -1) return -1;

    if(TEMP_FAILURE_RETRY( read(fd, buf, 12) ) != 12) goto outerr;
    if(memcmp(buf, leap_table_signature, 8)) {
        errno = EINVAL;
        return -1;
    }

#define GET_UINT32(_from) ( \
    ( ((uint8_t*)_from)[0] << 24 ) | \
    ( ((uint8_t*)_from)[1] << 16 ) | \
    ( ((uint8_t*)_from)[2] << 8 ) | \
    ( ((uint8_t*)_from)[3] ) \
)

    new_size = GET_UINT32(buf + 8);
    if((12 + new_size * 4) != st.st_size) {
        errno = EINVAL;
        goto outerr;
    }

    days = malloc(sizeof(int) * new_size);
    if(!days) goto outerr;

    for(i = 0; i < new_size; ++i) {
        if(TEMP_FAILURE_RETRY( read(fd, buf, 4) ) != 4) goto outerr;
        days[i] = GET_UINT32(buf);
        if(i && days[i] <= days[i - 1]) {
            errno = EINVAL;
            goto outerr;
        }
    }

    TEMP_FAILURE_RETRY( close(fd) );
    iso8601_leap_table_set(days, new_size);
    leap_table_free_old = 1;
    return 0;

#undef GET_UINT32

  outerr:
    saved_errno = errno;
    free(days);
    TEMP_FAILURE_RETRY( close(fd) );
    errno = saved_errno;
    return -1;
}



/* options for text editors
kate: replace-trailing-space-save true; space-indent true; tab-width 4;
vim: expandtab:ts=4:sw=4
*/
