/* libiso8601/src/libiso8601/100_types.h
 *
 *  (c)2006-2010, Laurence Withers, <l@lwithers.me.uk>.
 *  Released under the GNU GPLv3. See file COPYING or
 *  http://www.gnu.org/copyleft/gpl.html for details.
*/



/*! \brief Date/time point.

This structure contains the details to represent a specific instant on the UTC
timescale. It uses Jan 1, year 0000 as the origin (when \a day will be 0). \a
sec is the number of seconds elapsed since start of day, and \a nsec is the
number of nanoseconds elapsed since the start of the current second.

We correctly deal with leap seconds by encoding 23:59:60 as having a \a sec
field of 86400.

*/
struct iso8601_date {
    /*! \brief Number of nanoseconds elapsed since start of second. */
    int32_t nsec;

    /*! \brief Number of days elapsed since Jan 1, year 0000. May be negative.*/
    int32_t day;

    /*! \brief Number of seconds elapsed since start of day. */
    int32_t sec;
};



/*! \brief Date (day portion) precision.
\ingroup parser

This enumeration will record how precisely the date was specified, as well as
the format in use. It allows the library to determine the earliest and latest
dates that could possibly be represented with the given input and also allows
the output format to match the input format.

*/
enum iso8601_date_prec {
    /*! \brief Only year specified. */
    iso8601_prec_year,

    /*! \brief Year and month specified (calendar format). */
    iso8601_prec_month,

    /*! \brief Year, month and day specified (calendar format). */
    iso8601_prec_day,

    /*! \brief Year and ordinal day specified (ordinal format). */
    iso8601_prec_ord,

    /*! \brief Year and week specified (week format). */
    iso8601_prec_week,

    /*! \brief Year, week and weekday specified (week format). */
    iso8601_prec_wday
}date_prec;



/*! \brief Time precision.
\ingroup parser

This enumeration records how precisely the time was specified as well as its
format. The fractional format will record whether it was the hour, minute or
second that was specified with a fractional part, allowing a processed
date/time to be presented to the user in the format it was originally
encountered.

*/
enum iso8601_time_prec {
    /*! \brief Don't display date. */
    iso8601_prec_none,
    /*! \brief Display integer part of hour. */
    iso8601_prec_hour,
    /*! \brief Display hour and integer part of minute. */
    iso8601_prec_min,
    /*! \brief Display hour, minute and integer part of second. */
    iso8601_prec_sec,
    /*! \brief Display hour and fractional part of hour. */
    iso8601_prec_hourfrac,
    /*! \brief Display hour, minute and fractional part of minute. */
    iso8601_prec_minfrac,
    /*! \brief Display hour, minute, second and nanoseconds. */
    iso8601_prec_secfrac
}time_prec;



/*! \brief Date/time formatting details.

This structure simply records details related to the formatting (and precision)
of a date/time structure. The structure can be filled out by the parser so that
a program's output can match the format of its input. Alternatively it can be
controlled by the program to provide a consistent output format.

*/
struct iso8601_details {
    /*! \brief Date precision (\ref iso8601_date_prec). */
    uint8_t date_prec;

    /*! \brief Time precision (\ref iso8601_time_prec). */
    uint8_t time_prec;

    /*! \brief Flag: non-zero if extended format should be used. */
    uint8_t extended;

    /*! \brief Time zone offset in seconds. */
    int32_t tz_sec;
};



/*! \brief Short period elapsed time.

This structure contains the details of an elapsed time period, split into
seconds and nanoseconds.  None of its components can ever be negative. This is
only capable of representing short (compared to struct iso8601_date's range)
periods of up to approximately 136 years.

*/
struct iso8601_elapsed {
    /*! \brief Number of seconds. */
    uint32_t sec;
    /*! \brief Number of nanoseconds (0 to 999999999). */
    uint32_t nsec;
};



/* options for text editors
kate: replace-trailing-space-save true; space-indent true; tab-width 4;
vim: expandtab:ts=4:sw=4:syntax=ch.doxygen
*/
