/* libiso8601/src/libiso8601/200_parser.h
 *
 *  (c)2006-2010, Laurence Withers, <l@lwithers.me.uk>.
 *  Released under the GNU GPLv3. See file COPYING or
 *  http://www.gnu.org/copyleft/gpl.html for details.
*/



/*! \defgroup parser Parsing, printing and validation routines.

These routines are used for parsing an ISO8601 date/time string into the
internal structure used to represent them, and for validating such dates/times.

*/
/*!@{*/



/*! \brief Parse ISO8601 date/time.

\param str The input string. Whitespace will be stripped.
\param[out] earliest The earliest possible time the string could represent. May
    be 0.
\param[out] latest The latest possible time the string could represent. May be
    0.
\param[out] details Stores details such as the precision to which the time/date
    were specified. May be 0.
\retval -1 on error.
\retval 0 on success.

Parses a string containing the ISO8601 date/time. Deals with any format of
date, optionally storing the details in \a details. The time may be partial, in
which case this function returns the earliest and latest times that could
possibly be represented by the string.

Note that this function will accept leap seconds (23:59:60) on days on which
they occurred.

*/
int iso8601_parse(const char* str, struct iso8601_date* earliest,
    struct iso8601_date* latest, struct iso8601_details* details)
#ifndef DOXYGEN
    __attribute__((nonnull(1),warn_unused_result))
#endif
;



/*! \brief Print ISO8601 date/time into string.

\param str Pointer to buffer into which result is written.
\param amt Number of bytes in \a str.
\param date Date to print.
\param details Formatting details (may be 0).
\returns Pointer to buffer (\a str).

Formats and prints an ISO8601 date, optionally using the details in \a details.
Will always return a null-terminated result, even if that means truncating the
output to fit the buffer.

*/
char* iso8601_print(char* str, int amt, const struct iso8601_date* date, 
    const struct iso8601_details* details)
#ifndef DOXYGEN
    __attribute__((nonnull(1,3)))
#endif
;



/*! \brief Validate ISO8601 date/time.

\param date The date to validate.
\retval -1 if not valid.
\retval 0 if valid.

Checks the details of \a date to ensure that they are sensible. This involves
checking that \a sec is in the range 0 to 86399 (or 86400 if there is a leap
second), and that \a nsec is in the range 0 to 999999999.

*/
int iso8601_invalid(const struct iso8601_date* date)
#ifndef DOXYGEN
    __attribute__((nonnull))
#endif
;



/*!@}*/
/* options for text editors
kate: replace-trailing-space-save true; space-indent true; tab-width 4;
vim: expandtab:ts=4:sw=4:syntax=ch.doxygen
*/
