/* libiso8601/src/libiso8601/200_print.c
 *
 *  (c)2006-2010, Laurence Withers, <l@lwithers.me.uk>.
 *  Released under the GNU GPLv3. See file COPYING or
 *  http://www.gnu.org/copyleft/gpl.html for details.
*/



/* _default_details
 *  If the user doesn't pass precision details to iso8601_print(), these are
 *  used instead.
 */
static const struct iso8601_details
_default_details = {
    iso8601_prec_day,
    iso8601_prec_sec,
    1,
    0
};



char*
iso8601_print(char* str, int amt, const struct iso8601_date* date, 
    const struct iso8601_details* details)
{
    int y, m, d, ret = 0, extended;
    struct iso8601_date dttz;
    double frac;
    char* str_orig;
    struct iso8601_elapsed elapsed;

    str_orig = str;

    /* use default details if none provided */
    if(!details) details = &_default_details;

    /* adjust output for timezone */
    dttz = *date;
    if(details->tz_sec) {
        elapsed.sec = details ? details->tz_sec : 0;
        elapsed.nsec = 0;
        iso8601_add_elapsed(&dttz, &elapsed);
    }

    /* determine whether or not to force extended output */
    iso8601_to_cal(&y, &m, &d, &dttz);
    extended = details->extended || y < 0 || y > 9999;

    switch(details->date_prec) {
    case iso8601_prec_year:
        if(y < 0) snprintf(str, amt, "%05d", y);
        else snprintf(str, amt, "%04d", y);
        return str_orig;

    case iso8601_prec_month:
        if(y < 0) snprintf(str, amt, "%05d-%02d", y, m);
        else snprintf(str, amt, extended ? "%04d-%02d" : "%04d%02d", y, m);
        return str_orig;

    case iso8601_prec_day:
        if(y < 0) ret = snprintf(str, amt, "%05d-%02d-%02d", y, m, d);
        else ret = snprintf(str, amt, extended ?
            "%04d-%02d-%02d" : "%04d%02d%02d", y, m, d);
        break;

    case iso8601_prec_ord:
        iso8601_to_ord(&y, &d, &dttz);
        if(y < 0) ret = snprintf(str, amt, "%05d-%03d", y, d);
        else ret = snprintf(str, amt, extended ?
            "%04d-%03d" : "%04d%03d", y, d);
        break;

    case iso8601_prec_week:
        iso8601_to_week(&y, &m, &d, &dttz);
        /* note: ISO year can differ, so recompute extended flag */
        extended = y < 0 || y > 9999 || details->extended;
        if(y < 0) snprintf(str, amt, "%05d-W%02d", y, m);
        else snprintf(str, amt, extended ? "%04d-W%02d" : "%04dW%02d", y, m);
        return str_orig;

    case iso8601_prec_wday:
        iso8601_to_week(&y, &m, &d, &dttz);
        /* note: ISO year can differ, so recompute extended flag */
        extended = y < 0 || y > 9999 || details->extended;
        if(y < 0) ret = snprintf(str, amt, "%05d-W%02d-%d", y, m, d);
        else ret = snprintf(str, amt, extended ?
            "%04d-W%02d-%d" : "%04dW%02d%d", y, m, d);
        break;
    }

    if(ret < 1 || ret >= amt) return str_orig;
    str += ret;
    amt -= ret;

    switch(dttz.sec) {
    case 86400:
        y = 23;
        m = 59;
        d = 60;
        break;

    default:
        y = dttz.sec / 3600;
        dttz.sec -= y * 3600;
        m = dttz.sec / 60;
        dttz.sec -= m * 60;
        d = dttz.sec;
    }

    switch(details->time_prec) {
    case iso8601_prec_none:
        return str_orig;

    case iso8601_prec_hour:
        ret = snprintf(str, amt, "T%02d", y);
        break;

    case iso8601_prec_min:
        ret = snprintf(str, amt, extended ? "T%02d:%02d" : "T%02d%02d", y, m);
        break;

    case iso8601_prec_sec:
        ret = snprintf(str, amt, extended ?
            "T%02d:%02d:%02d" : "T%02d%02d%02d", y, m, d);
        break;

    case iso8601_prec_hourfrac:
        frac = y + m / 60.0 + (d + date->nsec / 1e9) / 3600.0;
        ret = snprintf(str, amt, "T%#012.9f", frac);
        break;

    case iso8601_prec_minfrac:
        frac = m + (d + date->nsec / 1e9) / 60.0;
        ret = snprintf(str, amt, extended ?
            "T%02d:%#012.9f" : "T%02d%#012.9f", y, frac);
        break;

    case iso8601_prec_secfrac:
        frac = d + date->nsec / 1e9;
        ret = snprintf(str, amt, extended ?
            "T%02d:%02d:%#012.9f" : "T%02d%02d%#012.9f", y, m, frac);
        break;
    }

    if(ret < 1 || (ret + 1) >= amt) return str_orig;
    str += ret;
    amt -= ret;

    if(details->tz_sec) {
        if(details->tz_sec < 0) {
            *str++ = '-';
            ret = -details->tz_sec;
        } else {
            *str++ = '+';
            ret = details->tz_sec;
        }
        --amt;

        y = ret / 3600;
        ret -= y * 3600;
        m = ret / 60;
        ret -= m * 60;
        d = ret;

        if(d) snprintf(str, amt, extended ?
            "%02d:%02d:%02d" : "%02d%02d%02d", y, m, d);
        else if(m) snprintf(str, amt, extended ?
            "%02d:%02d" : "%02d%02d", y, m);
        else snprintf(str, amt, "%02d", y);
    } else {
        *str++ = 'Z';
        *str++ = 0;
    }
    
    return str_orig;
}



/* options for text editors
kate: replace-trailing-space-save true; space-indent true; tab-width 4;
vim: expandtab:ts=4:sw=4
*/
