/* libutf8/src/libutf8/100_ctype.h
 *
 *  (c)2006-2009, Laurence Withers, <l@lwithers.me.uk>.
 *  Released under the GNU GPLv3. See file COPYING or
 *  http://www.gnu.org/copyleft/gpl.html for details.
*/



/*! \defgroup ctype Character classification

This module contains functions for character classification. These are
semantically equivalent to the \c is* functions defined in \c &lt;ctype.h&gt;,
except that they work on \c wchar_t UCS chars and are independent of the
system's current locale setting.

\todo There are many char classification functions that haven't been
      implemented yet. These will be added on demand.

*/
/*!@{*/



/*! \brief Test if character is ASCII.

\param ch Character to test.
\retval nonzero if \a ch is ASCII.
\retval 0 if \a ch is not ASCII.

This function tests a UCS char to see if it lies within the range of characters
that can be represented by ASCII (i.e. that the value of \a ch lies between 0
and 127, inclusive).

*/
int utf8_isascii(wchar_t ch);



/*! \brief Test if character is whitespace.

\param ch Character to test.
\retval nonzero if \a ch is whitespace.
\retval 0 if \a ch is not whitespace.

This function tests a UCS char to see if it should be classified as 
\ref char_class_whitespace "whitespace".

*/
int utf8_isspace(wchar_t ch);



/*! \brief Test if character is end-of-line.

\param ch Character to test.
\retval nonzero if \a ch is an EOL character.
\retval 0 if \a ch is not an EOL character.

This function tests a UCS char to see if it should be classified as 
\ref char_class_eol "end-of-line". Note that both ASCII LR and CF are treated
as EOL; it is up to the application to disambiguate the line ending in use.

*/
int utf8_iseol(wchar_t ch);



/*! \brief Test if character is valid UCS-4 codepoint.

\param ch The character to classify.
\retval nonzero If \a ch is a valid UCS-4 character.
\retval 0 If \a ch is not a valid UCS-4 character.

This function will examine a \c wchar_t value and determine whether or not it
is a valid UCS-4 character. Valid characters lie in the range
0&ndash;0x7FFFFFFF but exclude:
\li the UTF-16 surrogate code points (U+D800&ndash;U+DFFF, inclusive)
\li the invalid code points U+FFFE and U+FFFF

*/
int utf8_isucs4(wchar_t ch);



/*! \brief Test if character is valid UTF-32 (Unicode) codepoint.

\param ch The character to classify.
\retval nonzero If \a ch is a valid Unicode character.
\retval 0 If \a ch is not a valid Unicode character.

This function will examine a \c wchar_t value and determine whether or not it
is a valid Unicode character. Valid characters lie in the range
0&ndash;0x10FFFF but exclude:
\li the UTF-16 surrogate code points (U+D800&ndash;U+DFFF, inclusive)
\li the invalid code points U+FFFE and U+FFFF

*/
int utf8_isutf32(wchar_t ch);



/*! \brief Test if character is valid UTF-16 (Unicode) codepoint.

\param ch The character to classify.
\retval nonzero If \a ch is a valid Unicode character.
\retval 0 If \a ch is not a valid Unicode character.

This function will examine a \c wchar_t value and determine whether or not it
is a valid Unicode character that can be represented by a single UTF-16
codepoint. Valid characters lie in the range 0&ndash;0xFFFD but exclude:
\li the UTF-16 surrogate code points (U+D800&ndash;U+DFFF, inclusive)

*/
int utf8_isutf16(wchar_t ch);



/*!@}*/
/* options for text editors
kate: replace-trailing-space-save true; space-indent true; tab-width 4;
vim: expandtab:ts=4:sw=4:syntax=c.doxygen
*/
