/* libCStreamedXML/src/libCStreamedXML/200_types.h
 *
 *  (c)2006, Laurence Withers. Released under the GNU GPL. See file
 *  COPYING for more information / terms of license.
*/



/*! \brief Allocate and set up a new parser object.

\retval 0 on error (see \a errno).
\returns Pointer to newly-allocated parser object.

Call this function to obtain a parser context object. The object is initially populated with
sensible defaults (\a expandEntities is 1, the error callback functions print messages to stderr,
the data callback functions discard the data, etc.).

*/
struct csxml* csxml_newParser();



/*! \brief Free an existing parser object.

\param ctx The object to free (may be 0).

Frees a parser object and all of its associated data structures (strings, lists, etc.).

*/
void csxml_freeParser(struct csxml* ctx);



/*! \brief Parse a character.

\param ctx Parser context.
\param ch Character to parse.
\retval 0 on success.
\retval -1 on XML error.
\retval (non-0) if any callback function signals an error.

This function parses a single character. It will generate callbacks as appropriate. If one of the
callback functions returns a non-zero value, that value is returned from this function and the 
object is put into an error state. If the streamed XML is not well formed, or some other parsing 
error occurs, -1 will be returned and the object will be put into an error state.

Once in an error state, the function will simply discard data passed to it and return 0. However,
a lower-level parser is still running and is capable of matching stream restart markers, allowing
the parser to be reset.

*/
int csxml_feedChar(struct csxml* ctx, char ch);



/*! \brief Look up an entity.

\param ctx Parser context.
\param ent Entity name.
\retval 0 on error.
\returns Pointer to null-terminated string of expanded entity text.

This function is called to dereference entities. It handles the standard entities itself, and chains
on to the entityRef callback of the parser if needed. If the entity is not found, it calls the
unknownEntity callback and returns 0.

*/
const char* csxml_entityRef(struct csxml* ctx, const char* ent);



/*! \brief Reset parser.

\param ctx Parser context.

This function will reset the parser into its initial state, as though no data had yet been 
encountered. This is called after stream restart markers for example.

*/
void csxml_reset(struct csxml* ctx);




/*! \brief Parse a block of data.

\param ctx Parser context.
\param data Pointer to block of data.
\param amt Number of bytes to parse.

This function will parse each character in the block of data. If an error is encountered, the usual
process (callback, error state) is followed, but this function will continue to feed in data. The
idea is that you continue to feed in data from your data source, which might have a restart marker
(at which point the parsing will once again continue).

*/
void csxml_feedData(struct csxml* ctx, const char* data, size_t amt);

/* options for text editors
kate: replace-trailing-space-save true; space-indent true; tab-width 4;
vim: expandtab:ts=4:sw=4
*/
