/* libiso8601/src/libiso8601/400_calc.h
 *
 *  (c)2006-2010, Laurence Withers, <l@lwithers.me.uk>.
 *  Released under the GNU GPLv3. See file COPYING or
 *  http://www.gnu.org/copyleft/gpl.html for details.
*/



/*! \defgroup calc Conversion and calculation routines.

This set of functions is useful for converting dates and times into formats
understood by humans, and vice versa.

*/
/*!@{*/



/*! \brief Is year a leap year?

\param year Year to examine.
\retval 0 if not leap year.
\retval non-0 if leap year.

Returns non-0 if \a year is a leap year.

*/
int iso8601_isleap(int year);



/*! \brief Convert date to calendar format.

\param[out] year Calendar year.
\param[out] month Calendar month (1 to 12).
\param[out] day Calendar day (starting from 1).
\param date Date to convert.

*/
void iso8601_to_cal(int* year, int* month, int* day,
    const struct iso8601_date* date)
#ifndef DOXYGEN
    __attribute__((nonnull))
#endif
;



/*! \brief Convert date from calendar format.

\param[out] date Output date.
\param year Calendar year.
\param month Calendar month (1 to 12).
\param day Calenday day (starting from 1).
\retval 0 on success.
\retval -1 on error (and see \a errno).

Converts the date specified in \a year, \a month and \a day into \a date.

\note Does not touch the \a sec or \a nsec time members of \a date. This means
    they will be unchanged after a call to this function. If this will lead to
    unexpected results, initialise the structure to 0 first.

*/
int iso8601_from_cal(struct iso8601_date* date, int year, int month, int day)
#ifndef DOXYGEN
    __attribute__((nonnull))
#endif
;



/*! \brief Convert date to ordinal format.

\param[out] year Calendar year.
\param[out] oday Ordinal day (from 1).
\param date Date to convert.

*/
void iso8601_to_ord(int* year, int* oday, const struct iso8601_date* date)
#ifndef DOXYGEN
    __attribute__((nonnull))
#endif
;



/*! \brief Convert date from ordinal format.

\param[out] date Output date.
\param year Calendar year.
\param oday Ordinal day (from 1).
\retval 0 on success.
\retval -1 on error (and see \a errno).

Converts the date specified into \a year and \a oday into \a date.

\note Does not touch the \a sec or \a nsec time members of \a date. This means
    they will be unchanged after a call to this functoin. If this will lead to
    unexpected results, initialise the structure to 0 first.

*/
int iso8601_from_ord(struct iso8601_date* date, int year, int oday)
#ifndef DOXYGEN
    __attribute__((nonnull))
#endif
;



/*! \brief Convert date to week format.

\param[out] year Calendar year.
\param[out] week ISO week number (from 1).
\param[out] wday ISO week day number (1 = Monday, 7 = Sunday).
\param date Date to convert.

*/
void iso8601_to_week(int* year, int* week, int* wday,
    const struct iso8601_date* date)
#ifndef DOXYGEN
    __attribute__((nonnull))
#endif
;



/*! \brief Convert date from week format.

\param [out] date Output date.
\param year Calendar year.
\param week ISO week number (from 1).
\param wday ISO week day number (1 = Monday, 7 = Sunday).
\retval 0 on success.
\retval -1 on error (and see \a errno).

Converts the date specified into \a year, \a week and \a wday into \a date.

\note Does not touch the \a sec or \a nsec time members of \a date. This means
    they will be unchanged after a call to this functoin. If this will lead to
    unexpected results, initialise the structure to 0 first.

*/
int iso8601_from_week(struct iso8601_date* date, int year, int week, int wday)
#ifndef DOXYGEN
    __attribute__((nonnull))
#endif
;



/*! \brief Convert time to wall clock format.

\param[out] hour Output hour.
\param[out] min Output minute.
\param[out] sec Output second.
\param date Date/time to convert.

Converts the time stored in \a date into wall clock format, storing the result
in \a hour, \a min and \a sec.

*/
void iso8601_to_clocktime(int* hour, int* min, int* sec,
    const struct iso8601_date* date)
#ifndef DOXYGEN
    __attribute__((nonnull))
#endif
;



/*! \brief Convert time from wall clock format.

\param[out] date Output time.
\param hour Hour.
\param min Minute.
\param sec Second.
\retval 0 on success.
\retval -1 on error (and see \a errno).

Converts the time as specified by \a hour, \a min and \a sec, storing the
result in \a date.

\note Does not touch the \a day (date) member of \a date. This means it will be
    unchanged after a call to this function. If this will lead to unexpected
    results, initialise the structure to 0 first.

*/
int iso8601_from_clocktime(struct iso8601_date* date, int hour, int min,
    int sec)
#ifndef DOXYGEN
    __attribute__((nonnull))
#endif
;



/*!@}*/
/* options for text editors
kate: replace-trailing-space-save true; space-indent true; tab-width 4;
vim: expandtab:ts=4:sw=4:syntax=ch.doxygen
*/
