#!/bin/bash
# check-revdep.sh
#
#  Script to check reverse dependencies. The machine filesystem to check is specified as
#  BUILD_MACHINE. If not set, we cycle through all known machines, recursively calling this script.
#
#  For each ELF object in the machine's filesystem, we check that its shared library dependencies
#  are satisfied. If not, we find the package associated with the file and then rebuild the package.
#



# INITIAL PHASE ####################################################################################
#                                                                                                  #
#  If we are not performing a specific check (BUILD_MACHINE), recurse. Otherwise, fall through.    #
#                                                                                                  #
####################################################################################################



# automatically exit on failure
set -e

# check for unwanted recursion
[ -n "${REVDEP_DO_NOT_RECURSE}" ] && exit 0

# read all configuration files
[ -e "conf.local" ] && source "./conf.local"
source ./machine.conf

# if BUILD_MACHINE is not set, recursively build for all machines
if [ -z "${BUILD_MACHINE}" ]
then
    for MACH in ${MACHINE_LIST}
    do
        BUILD_MACHINE="${MACH}" ./check-revdep.sh
    done

    exit 0
fi



# REVERSE DEPENDENCY CHECK #########################################################################
#                                                                                                  #
#  Once the filesystem merge is complete, we must check reverse dependencies. SO_CHANGED will be   #
#  non-zero if any shared libraries were changed.                                                  #
#                                                                                                  #
####################################################################################################

echo "Checking reverse dependencies for machine \"${BUILD_MACHINE}\"..."
machine_setup "${BUILD_MACHINE}"

# now do a reverse dependency check on every library and executable
TMP_BADLIST="$(mktemp)"
TMP_FILELIST="$(mktemp)"
TMP_DEPLIST="$(mktemp)"
cd "filesystems/${BUILD_MACHINE}"
if [ "${ARCH_HAS_USRMOVE}" -eq 0 ]
then
    [ -d "sbin" ] && find "sbin/" -type f >> "${TMP_FILELIST}"
    [ -d "bin" ] && find "bin/" -type f >> "${TMP_FILELIST}"
    [ -d "lib" ] && find "lib/" -type f >> "${TMP_FILELIST}"
    [ -d "srv/http/cgi-bin" ] && find "srv/http/cgi-bin" -type f >> "${TMP_FILELIST}"
    [ -d "srv/http/cgi-bin.auth" ] && find "srv/http/cgi-bin.auth" -type f >> "${TMP_FILELIST}"
else
    [ -d "usr/srv/http/cgi-bin" ] && find "usr/srv/http/cgi-bin" -type f >> "${TMP_FILELIST}"
    [ -d "usr/srv/http/cgi-bin.auth" ] && find "usr/srv/http/cgi-bin.auth" -type f >> "${TMP_FILELIST}"
fi
[ -d "usr/sbin" ] && find "usr/sbin/" -type f >> "${TMP_FILELIST}"
[ -d "usr/bin" ] && find "usr/bin/" -type f >> "${TMP_FILELIST}"
[ -d "usr/lib" ] && find "usr/lib/" -type f >> "${TMP_FILELIST}"

# for each file, if it is a shared library or dynamic executable, record its dependencies
while read -r FILE
do
    case "$(file "${FILE}")" in
    *"shared object"* | *executable*dynamic* )
        ;;

    **)
        continue
        ;;
    esac

    readelf --dynamic "${FILE}" | \
        grep "NEEDED.*Shared library" | \
        sed -e s,'^.*\[\(.*\)\].*$,\1,' \
        > "${TMP_DEPLIST}"

    # check that each dependency is fulfilled
    while read -r DEP_SONAME
    do
        if [ ! -e "lib/${DEP_SONAME}" -a ! -e "usr/lib/${DEP_SONAME}" ]
        then
            echo "${FILE}" >> "${TMP_BADLIST}"
            echo " * \"${FILE}\" is broken (requires missing \"${DEP_SONAME}\")"
            break
        fi
    done < "${TMP_DEPLIST}"
done < "${TMP_FILELIST}"

# assign files in TMP_BADLIST to packages
cd "../.."
TMP_PKGLIST="$(mktemp)"
while read -r FILE
do
    ASSIGNED="0"
    while read -r PKG
    do
        FILELIST="crosslib/${BUILD_MACHINE}/share/filelist/${PKG}"
        [ ! -f "${FILELIST}" ] && continue
        if grep -q "^./${FILE}\$" "${FILELIST}"
        then
            echo "${PKG}" >> "${TMP_PKGLIST}"
            ASSIGNED="1"
            break
        fi
        if grep -q "^./${FILE/lib/lib64}\$" "${FILELIST}"
        then
            echo "${PKG}" >> "${TMP_PKGLIST}"
            ASSIGNED="1"
            break
        fi
    done < "packages/list"

    if [ "${ASSIGNED}" -eq 0 ]
    then
        echo "File \"${FILE}\" has broken dependencies but cannot be assigned to package."
        ABORT="1"
    fi
done < "${TMP_BADLIST}"

[ -n "${ABORT}" ] && exit 1

# rebuild packages in dependency order
TMP_SORTEDLIST="$(mktemp)"
cat "${TMP_PKGLIST}" | sort | uniq > "${TMP_SORTEDLIST}"
export BUILD_MACHINE
export REVDEP_DO_NOT_RECURSE="1"
while read -r PKG
do
    grep -q -- "^${PKG}\$" "${TMP_SORTEDLIST}" || continue
    FORCE_FRESH_BUILD="1" BUILD_PACKAGE="${PKG}" ./build.sh
done < "packages/list"

# clean up
rm "${TMP_BADLIST}" "${TMP_FILELIST}" "${TMP_DEPLIST}" "${TMP_PKGLIST}" "${TMP_SORTEDLIST}"

# vim: ts=4:sw=4:expandtab
