/* libiso8601/src/libiso8601/400_manip.c
 *
 *  (c)2006-2010, Laurence Withers, <l@lwithers.me.uk>.
 *  Released under the GNU GPLv3. See file COPYING or
 *  http://www.gnu.org/copyleft/gpl.html for details.
*/



int
iso8601_lt(const struct iso8601_date* d1, const struct iso8601_date* d2)
{
    if(d1->day < d2->day) return 1;
    if(d1->day > d2->day) return 0;
    if(d1->sec < d2->sec) return 1;
    if(d1->sec > d2->sec) return 0;
    if(d1->nsec < d2->nsec) return 1;
    return 0;
}



int
iso8601_lte(const struct iso8601_date* d1, const struct iso8601_date* d2)
{
    if(d1->day < d2->day) return 1;
    if(d1->day > d2->day) return 0;
    if(d1->sec < d2->sec) return 1;
    if(d1->sec > d2->sec) return 0;
    if(d1->nsec < d2->nsec) return 1;
    return d1->nsec == d2->nsec;
}



int
iso8601_eq(const struct iso8601_date* d1, const struct iso8601_date* d2)
{
    return (d1->day == d2->day) && (d1->sec == d2->sec)
        && (d1->nsec == d2->nsec);
}



int
iso8601_cmp(const struct iso8601_date* d1, const struct iso8601_date* d2)
{
    if(d1->day < d2->day) return -1;
    if(d1->day > d2->day) return 1;
    if(d1->sec < d2->sec) return -1;
    if(d1->sec > d2->sec) return 1;
    if(d1->nsec < d2->nsec) return -1;
    if(d1->nsec > d2->nsec) return 1;
    return 0;
}



void
iso8601_add_elapsed(struct iso8601_date* date,
    const struct iso8601_elapsed* per)
{
    div_t qr;
    int leapcorrect;

    /* work out number to advance ‘day’ by */
    qr = div(per->sec, 86400);
    leapcorrect = _leap_elapsed_day(date->day, date->day + qr.quot);
    date->day += qr.quot;
    date->sec += qr.rem - leapcorrect;

    date->nsec += per->nsec;
    if(date->nsec >= BILLION) {
        ++date->sec;
        date->nsec -= BILLION;
    }

    if(date->sec < 0) {
        --date->day;
        date->sec += iso8601_seconds_leap(date);
    }

    if(date->sec >= iso8601_seconds_leap(date)) {
        date->sec -= iso8601_seconds_leap(date);
        ++date->day;
    }
}



void 
iso8601_subtract_elapsed(struct iso8601_date* date,
    const struct iso8601_elapsed* per)
{
    div_t qr;
    int leapcorrect;

    /* work out number to advance ‘day’ by */
    qr = div(per->sec, 86400);
    leapcorrect = _leap_elapsed_day(date->day - qr.quot, date->day);
    date->day -= qr.quot;
    date->sec -= qr.rem - leapcorrect;
    date->nsec -= per->nsec;

    if(date->nsec < 0) {
        --date->sec;
        date->nsec += BILLION;
    }

    if(date->sec < 0) {
        --date->day;
        date->sec += iso8601_seconds_leap(date);
    }

    if(date->sec >= iso8601_seconds_leap(date)) {
        date->sec -= iso8601_seconds_leap(date);
        ++date->day;
    }
}



void
iso8601_add_multiple(struct iso8601_date* date,
    const struct iso8601_elapsed* per, int n)
{
    long long nsec;
    lldiv_t qr;
    struct iso8601_elapsed mult;

    nsec = per->nsec * llabs(n);
    qr = lldiv(nsec, BILLION);
    mult.sec = qr.quot + per->sec * llabs(n);
    mult.nsec = qr.rem;

    if(n < 0) iso8601_subtract_elapsed(date, &mult);
    else iso8601_add_elapsed(date, &mult);
}



void
iso8601_difference(const struct iso8601_date* d1, const struct iso8601_date* d2,
    struct iso8601_elapsed* per, int* sign)
{
    const struct iso8601_date* e1, * e2;

    /* compute sign */
    if(iso8601_lt(d1, d2)) {
        if(sign) *sign = -1;
        e1 = d2;
        e2 = d1;
    } else {
        if(sign) *sign = 1;
        e1 = d1;
        e2 = d2;
    }

    if(!per) return;

    /* compute difference, knowing that e1 >= e2 */
    per->sec = (e1->day - e2->day) * 86400
        + iso8601_leap_elapsed(e2, e1 /* start, end */)
        + (e1->sec - e2->sec);
    if(e1->nsec < e2->nsec) {
        --per->sec;
        per->nsec = BILLION + e1->nsec - e2->nsec;
    } else {
        per->nsec = e1->nsec - e2->nsec;
    }
}



int
iso8601_elapsed_div(const struct iso8601_elapsed* num,
    const struct iso8601_elapsed* denom, struct iso8601_elapsed* remain)
{
    unsigned long long pnum, pdenom;
    lldiv_t val, v2;

    pnum = num->sec;
    pnum *= BILLION;
    pnum += num->nsec;

    pdenom = denom->sec;
    pdenom *= BILLION;
    pdenom += denom->nsec;

    val = lldiv(pnum, pdenom);
    if(remain) {
        v2 = lldiv(val.rem, BILLION);
        remain->sec = v2.quot;
        remain->nsec = v2.rem;
    }
    return val.quot;
}



/* options for text editors
kate: replace-trailing-space-save true; space-indent true; tab-width 4;
vim: expandtab:ts=4:sw=4
*/
