/* libiso8601/src/libiso8601/100_leap.c
 *
 *  (c)2006, Laurence Withers, <l@lwithers.me.uk>.
 *  Released under the GNU GPLv2. See file COPYING or
 *  http://www.gnu.org/copyleft/gpl.html for details.
*/



static int leap_second_days_table[] = {
    720439, /* 1972-06-30 */
    720623, /* 1972-12-31 */
    720988, /* 1973-12-31 */
    721353, /* 1974-12-31 */
    721718, /* 1975-12-31 */
    722084, /* 1976-12-31 */
    722449, /* 1977-12-31 */
    722814, /* 1978-12-31 */
    723179, /* 1979-12-31 */
    723726, /* 1981-06-30 */
    724091, /* 1982-06-30 */
    724456, /* 1983-06-30 */
    725187, /* 1985-06-30 */
    726101, /* 1987-12-31 */
    726832, /* 1989-12-31 */
    727197, /* 1990-12-31 */
    727744, /* 1992-06-30 */
    728109, /* 1993-06-30 */
    728474, /* 1994-06-30 */
    729023, /* 1995-12-31 */
    729570, /* 1997-06-30 */
    730119, /* 1998-12-31 */
    732676, /* 2005-12-31 */
    733772, /* 2008-12-31 */
};



static int* leap_second_days = leap_second_days_table;
static int leap_second_days_num = sizeof(leap_second_days_table) / sizeof(int);



int iso8601_seconds_leap(const struct iso8601_date* date)
{
    int i;
    for(i = 0; i < leap_second_days_num; ++i) if(leap_second_days[i] == date->day) return 86401;
    return 86400;
}



static int _leap_elapsed_day(int sday, int eday)
{
    int spos, epos;

    for(spos = 0; spos < leap_second_days_num; ++spos) {
        if(sday <= leap_second_days[spos]) break;
    }
    for(epos = 0; epos < leap_second_days_num; ++epos) {
        if(eday <= leap_second_days[epos]) break;
    }

    return epos - spos;
}



int iso8601_leap_elapsed(const struct iso8601_date* start, const struct iso8601_date* end)
{
    return _leap_elapsed_day(start->day, end->day);
}



static int leap_table_free_old = 0;

void iso8601_leap_table_set(int* new_table, int new_size)
{
    if(leap_table_free_old) free(leap_second_days);
    leap_table_free_old = 0;

    leap_second_days = new_table;
    leap_second_days_num = new_size;
}



static const char* leap_table_signature = "/O9PdPZI";



int iso8601_leap_table_load(const char* fname)
{
    struct stat st;
    int fd, saved_errno, i, new_size, * days = 0;
    char buf[12];

    if(!fname) fname = DEFAULT_LEAP_TABLE;

    if(stat(fname, &st)) return -1;
    if(st.st_size < 12) {
        errno = EINVAL;
        return -1;
    }

    fd = open(fname, O_RDONLY);
    if(fd == -1) return -1;

    if(TEMP_FAILURE_RETRY( read(fd, buf, 12) ) != 12) goto outerr;
    if(memcmp(buf, leap_table_signature, 8)) {
        errno = EINVAL;
        return -1;
    }

#define GET_UINT32(_from) ( \
    ( ((uint8_t*)_from)[0] << 24 ) | \
    ( ((uint8_t*)_from)[1] << 16 ) | \
    ( ((uint8_t*)_from)[2] << 8 ) | \
    ( ((uint8_t*)_from)[3] ) \
)

    new_size = GET_UINT32(buf + 8);
    if((12 + new_size * 4) != st.st_size) {
        errno = EINVAL;
        goto outerr;
    }

    days = malloc(sizeof(int) * new_size);
    if(!days) goto outerr;

    for(i = 0; i < new_size; ++i) {
        if(TEMP_FAILURE_RETRY( read(fd, buf, 4) ) != 4) goto outerr;
        days[i] = GET_UINT32(buf);
        if(i && days[i] <= days[i - 1]) {
            errno = EINVAL;
            goto outerr;
        }
    }

    TEMP_FAILURE_RETRY( close(fd) );
    iso8601_leap_table_set(days, new_size);
    leap_table_free_old = 1;
    return 0;

#undef GET_UINT32

  outerr:
    saved_errno = errno;
    free(days);
    TEMP_FAILURE_RETRY( close(fd) );
    errno = saved_errno;
    return -1;
}



/* options for text editors
kate: replace-trailing-space-save true; space-indent true; tab-width 4;
vim: expandtab:ts=4:sw=4
*/
