#!/bin/bash

# Try to determine which packages have been modified and build only them

packages_checked=0
packages_built=0
packages_failed=0


# Select an input file checking for a machine specific version.
# Process it stripping blank lines and # comments and present the remainder
# to stdout.  You get nothing if the file don't exist

select_and_cstrip() {
	local file="$1"

	[ -e "${file}-${BUILD_MACHINE}" ] && file="${file}-${BUILD_MACHINE}"
	
    	[ -e "$file" ] && sed -e 's/#.*//' -e '/^[	 ]*$/d' $file
}

# Expand a list of file and directory names
expand_list() {
	local incdir=""
	local i

	# -d flag includes directories in the output
	if [ "$1" = "-d" ]
	then
		incdir="-o -type d"
		shift
	fi

	for i in $*
	do
		if [ -n "$i" ]
		then
			if [ -d "$i" ]
			then
				find "$i" \( -type f $incdir \) -print
			elif [ -e "$i" ]
			then
				echo "$i"
			fi
		fi
	done
}


find_older_newest() {
	local test=$1
	local dflag="$2"
	local file
	local res=""
	shift ; shift

	while read file
	do
		[ -z "$res" ] && res="$file"
		[ -e "$file" -a "$file" $test "$res" ] && res="$file"
	done < <(expand_list $dflag $* )

	[ -n "$res" ] && echo "$res"
}

find_newest() {
	find_older_newest -nt -d $*
}

find_oldest() {
	find_older_newest -ot "" $*
}

call_build() {
	local result

	packages_built=$(expr $packages_built + 1)

	BUILD_MACHINE=$BUILD_MACHINE BUILD_PACKAGE=$BUILD_PACKAGE ./build.sh
	result=$?
	if [ "$result" != "0" ]
	then
	    packages_failed=$(expr $packages_failed + 1)
	fi
}

check_build_package() {
	BUILD_PACKAGE=$1
	local dst
	local src
	local dlist="packages/$BUILD_PACKAGE/Buildroot/$BUILD_MACHINE"
	local slist="`echo packages/$BUILD_PACKAGE/{patches,static,initramfs}`"
	local record_build=n
	local file

	# Add plain files to the source list with the exception of Log
	for file in packages/$BUILD_PACKAGE/*
	do
		if [ -f $file ]
		then
			case $file in
			*/Log)							;;
			*)	slist="$slist $file"	;;
			esac
		fi
	done

	# Modify lists for some packages
	case "$BUILD_PACKAGE" in
	glibc)
		# Use a marker file as the original dates are conserved
		dlist="packages/$BUILD_PACKAGE/.build_marker.$BUILD_MACHINE"
		record_build=y
		;;
#	mk4-kernel)
#		# include initramfs source files for flash target
#		slist="$slist filesystems/$BUILD_MACHINE"
#		;;
	mk4-system)
		# include DB defaults files used to build first DB
		slist="$slist filesystems/$BUILD_MACHINE/etc/configdb/defaults/"
		;;
	*)	;;
	esac

	# Packages with static copies files suffer from the original dates being
	# preserved problem as glibc and we apply the same marker file solution
	if [ -d "packages/$BUILD_PACKAGE/static" ]
	then
		dlist="packages/$BUILD_PACKAGE/.build_marker.$BUILD_MACHINE"
		record_build=y
	fi

	if [ ! -d "packages/$BUILD_PACKAGE/Buildroot/$BUILD_MACHINE" ]
	then
		call_build
	elif [ ! -e "$dlist" ]
	then
		call_build
	else
		dst=`eval find_oldest $dlist`
		[ -z "$dst" ] && dst="$dlist"
		src=`eval find_newest $slist`
		if [ "$src" -nt "$dst" ]
		then
			echo Building $BUILD_PACKAGE because $src newer than $dst
			call_build
		else
			record_build=n
		fi
	fi

	if [ $record_build = y ]
	then
		date >> packages/$BUILD_PACKAGE/.build_marker.$BUILD_MACHINE
	fi

	packages_checked=$(expr $packages_checked + 1)
}

check_build_machine() {
	local pkg
	BUILD_MACHINE=$1

	while read pkg
	do
		check_build_package $pkg
	done < <( select_and_cstrip packages/list )
}

if [ ! -x ./build.sh -o ! -d packages ]
then
	echo "Needs to be run in ~/git/platinum-build/"
	exit 1
fi

check_build_machine CMG-DCM-mk4

echo
echo "Packages checked: $packages_checked"
echo "Packages built:   $packages_built"
echo "Packages failed:  $packages_failed"

# Set exit code
[ "$packages_failed" = "0" ]
