/* libiso8601/src/libiso8601/400_manip.h
 *
 *  (c)2006, Laurence Withers, <l@lwithers.me.uk>.
 *  Released under the GNU GPLv2. See file COPYING or
 *  http://www.gnu.org/copyleft/gpl.html for details.
*/



/*! \defgroup manip Manipulation routines.

This set of functions is useful for performing arithmetic etc. on dates. It uses 
struct iso8601_elapsed to represent the magnitude of time differences.

*/
/*!@{*/



/*! \brief Comparison (less than).

\param d1 First date to compare.
\param d2 Second date to compare.
\retval non-0 if \a d1 &lt; \a d2
\retval 0 otherwise

*/
int iso8601_lt(const struct iso8601_date* d1, const struct iso8601_date* d2);



/*! \brief Comparison (less than or equal to).

\param d1 First date to compare.
\param d2 Second date to compare.
\retval non-0 if \a d1 &lt;= \a d2
\retval 0 otherwise

*/
int iso8601_lte(const struct iso8601_date* d1, const struct iso8601_date* d2);



/*! \brief Comparison (equality).

\param d1 First date to compare.
\param d2 Second date to compare.
\retval non-0 if \a d1 == \a d2
\retval 0 otherwise

*/
int iso8601_eq(const struct iso8601_date* d1, const struct iso8601_date* d2);



/*! \brief Comparison (for qsort et al.).

\param d1 First date to compare.
\param d2 Second date to compare.
\retval -1 if \a d1 &lt; \a d2
\retval 0 if \a d1 == \a d2
\retval 1 if \a d1 &gt; \a d2

*/
int iso8601_cmp(const struct iso8601_date* d1, const struct iso8601_date* d2);



/*! \brief Add a period to a date.

\param[in,out] date Date to modify.
\param per Period to advance date/time by.

*/
void iso8601_add_elapsed(struct iso8601_date* date, const struct iso8601_elapsed* per);



/*! \brief Subtract a period from a date.

\param[in,out] date Date to modify.
\param per Period to regress date/time by.

*/
void iso8601_subtract_elapsed(struct iso8601_date* date, const struct iso8601_elapsed* per);



/*! \brief Add a multiple of a period to a date.

\param[in,out] date Date to modify.
\param per Period to advance date/time by.
\param n Multiple of \a per.

Adds \a n multiples of \a per to \a date. \a n may be 0 or negative. The result is stored in
\a date. This is an efficient implementation which avoids loops, but it does use 64-bit arithmetic.

*/
void iso8601_add_multiple(struct iso8601_date* date, const struct iso8601_elapsed* per, int n);



/*! \brief Find difference between dates.

\param d1 First date.
\param d2 Second date.
\param[out] per Magnitude of period elapsed between two dates. Pointer may be 0.
\param[out] sign Set to sign of difference (-1 or +1). Pointer may be 0.

This function will perform the calculation <code>|d1 - d2|</code>, storing the result in \a per (if
it is not a null pointer). The sign of the result is stored in \a sign (if it is not a null 
pointer), i.e. -1 if \a d2 &gt; \a d1 or +1 if \a d2 &lt;= \a d1.

*/
void iso8601_difference(const struct iso8601_date* d1, const struct iso8601_date* d2,
    struct iso8601_elapsed* per, int* sign);



/*! \brief Divide one period by another.

\param num Numerator.
\param denom Denominator (divisor).
\param[out] remain Remainder. May be 0.
\returns Number of times \a denom divides into \a num.

This function computes the number of times that \a denom can be divided into \a num, returning that
number. If desired, the remaining period which could not be divided can be written into \a remain.
Uses 64-bit arithmetic internally.

*/
int iso8601_elapsed_div(const struct iso8601_elapsed* num, const struct iso8601_elapsed* denom,
    struct iso8601_elapsed* remain)
#ifndef DOXYGEN
    __attribute__((nonnull(1,2)));
#endif
;



/*!@}*/
/* options for text editors
kate: replace-trailing-space-save true; space-indent true; tab-width 4;
vim: expandtab:ts=4:sw=4:syntax=ch.doxygen
*/
