/* libiso8601/src/libiso8601/400_leap.h
 *
 *  (c)2006, Laurence Withers, <l@lwithers.me.uk>.
 *  Released under the GNU GPLv2. See file COPYING or
 *  http://www.gnu.org/copyleft/gpl.html for details.
*/



/*! \defgroup leap Leap second support.

A set of functions for explicitly dealing with leap seconds. All library functions are implicitly
leap second aware, but there are occasions when leap seconds must be dealt with explicitly. These
functions permit this.

Internally, leap seconds are represented as a table of day numbers; each day number present in the
table means that the corresponding day has 86401 seconds (a leap second).

\note The library does not (yet) have support for negative leap seconds. This support will be added
    should a day with a negative leap second ever come. This will need to addressed in the API as
    the current API does not support loading a table of negative leap seconds.

It is possible to create a disk file containing an updated table and to load that file using
\ref iso8601_leap_table_load (or to explicitly use an existing in-memory table with
\ref iso8601_leap_table_set). The format of the on-disk file is:

<pre># integers are stored big-endian
[8*char] signature, "/O9PdPZI"
[uint32] number of entries, 'n'

# entries are repeated 'n' times, and stored lowest day number first
[uint32] day number of entry</pre>

*/
/*!@{*/



/*! \brief Return number of seconds in day, taking leap seconds into account.

\param date The date to return the number of seconds for.
\retval 86399 day with a negative leap second.
\retval 86400 day with no leap second.
\retval 86401 day with one leap second.

Returns the duration of a day \a date, in seconds. This function takes leap seconds into account and
may be used to determine if a day contains a leap second or not.

\note There have not yet been any days requiring a negative leap second, so at present 86399 will
    never be returned.

*/
int iso8601_seconds_leap(const struct iso8601_date* date);



/*! \brief Return number of leap seconds elapsed between two days.

\param start The start date.
\param end The end date.
\returns Number of leap seconds elapsed.

Computes the number of leap seconds that have elapsed between two days. Note that this is the sum of
such leap seconds, so it will be 0 if (for example) there is one positive leap second and one 
negative leap second. The ordering of the dates is important; if \a start is after \a end, then the
value returned will be negative (for positive leap seconds).

*/
int iso8601_leap_elapsed(const struct iso8601_date* start, const struct iso8601_date* end);



/*! \brief Update table of leap seconds.

\param new_table Array of day numbers on which leap seconds occur.
\param new_size Number of entries in array.

This function can be used to update the table of leap seconds at runtime. The \a new_table argument
points to an array of integers, each entry being the day number containing a leap second.  The array
must be sorted so that lower day numbers appear towards the start of the array. \a new_size gives
the number of entries in the array. \a new_table must persist in memory as long as library functions
are in use.

\warning If negative leap seconds are ever used, this function will not support them. There may need
    to be an ABI change in future to solve this problem.

*/
void iso8601_leap_table_set(int* new_table, int new_size) __attribute__((nonnull));



/*! \brief Load new table of leap seconds from disk.

\param fname Filename. May be 0 for system default.
\retval 0 on success.
\retval -1 on error (and see \a errno).

This function attempts to load a new table of leap seconds from disk, using the filename \a fname.
If \a fname is not specified, the system default (set at compile time) will be used.

If the table is loaded successfully, it will be set with \ref iso8601_leap_table_set(). On any
error, -1 will be returned, and \a errno will be set appropriately. If \a errno is \c EINVAL, then
the file does not contain a valid leap second table.

*/
int iso8601_leap_table_load(const char* fname);



/*!@}*/
/* options for text editors
kate: replace-trailing-space-save true; space-indent true; tab-width 4;
vim: expandtab:ts=4:sw=4:syntax=ch.doxygen
*/
