/* libiso8601/src/libiso8601/400_c_library.c
 *
 *  (c)2006, Laurence Withers, <l@lwithers.me.uk>.
 *  Released under the GNU GPLv2. See file COPYING or
 *  http://www.gnu.org/copyleft/gpl.html for details.
*/

// days between 0000-001 and 1970-001 (the unix epoch)
#define EPOCH_GDAY (719528)



void iso8601_now(struct iso8601_date* date, struct iso8601_details* details)
{
    static int use_gettimeofday = 0;
    struct timespec ts;
    struct timeval tv;
    struct tm tm;

    /* not all systems have clock_gettime implemented */
    if(use_gettimeofday || clock_gettime(CLOCK_REALTIME, &ts)) {
        use_gettimeofday = 1;
        gettimeofday(&tv, 0);
        ts.tv_sec = tv.tv_sec;
        ts.tv_nsec = tv.tv_usec * 1000L;
    }

    /* populate the `struct iso8601_date' if it was passed */
    if(date) {
        iso8601_from_ts(date, &ts);
    }

    /* populate the `struct iso8601_details' if it was passed */
    if(details) {
        memset(details, 0, sizeof(struct iso8601_details));
        details->date_prec = iso8601_prec_day;
        details->time_prec = iso8601_prec_secfrac;

        /* this is silly, but it's the only way to recover the timezone */
        localtime_r(&ts.tv_sec, &tm);
        details->tz_sec = tm.tm_gmtoff;
    }
}



int iso8601_set_sysclock(const struct iso8601_date* date)
{
    static int use_settimeofday = 0;
    struct timeval tv;
    struct timespec ts;

    /* not all systems have clock_settime implemented */
    if(use_settimeofday) goto try_settimeofday;

    /* try setting clock */
    iso8601_to_ts(&ts, date);
    if(clock_settime(CLOCK_REALTIME, &ts)) {
        if(errno == EPERM) return -1;
        use_settimeofday = 1;
        goto try_settimeofday;
    }
    
    return 0;

    /* fallback using settimeofday() */
  try_settimeofday:
    iso8601_to_tv(&tv, date);
    return settimeofday(&tv, 0);
}



void iso8601_from_ts(struct iso8601_date* date, const struct timespec* ts)
{
    ldiv_t qr;

    qr = ldiv(ts->tv_sec, 86400);
    date->day = EPOCH_GDAY + qr.quot;
    date->sec = qr.rem;
    date->nsec = ts->tv_nsec;
}



void iso8601_to_ts(struct timespec* ts, const struct iso8601_date* date)
{
    ts->tv_sec = 86400L * (date->day - EPOCH_GDAY) + date->sec;
    ts->tv_nsec = date->nsec;
}



void iso8601_from_tv(struct iso8601_date* date, const struct timeval* tv)
{
    ldiv_t qr;

    qr = ldiv(tv->tv_sec, 86400);
    date->day = EPOCH_GDAY + qr.quot;
    date->sec = qr.rem;
    date->nsec = tv->tv_usec * 1000;
}



void iso8601_to_tv(struct timeval* tv, const struct iso8601_date* date)
{
    tv->tv_sec = 86400L * (date->day - EPOCH_GDAY) + date->sec;
    tv->tv_usec = date->nsec / 1000;
}



void iso8601_from_time_t(struct iso8601_date* date, const time_t* t)
{
    ldiv_t qr;

    qr = ldiv(*t, 86400);
    date->day = EPOCH_GDAY + qr.quot;
    date->sec = qr.rem;
    date->nsec = 0;
}



void iso8601_to_time_t(time_t* t, const struct iso8601_date* date)
{
    *t = 86400L * (date->day - EPOCH_GDAY) + date->sec;
}



/* options for text editors
kate: replace-trailing-space-save true; space-indent true; tab-width 4;
vim: expandtab:ts=4:sw=4
*/
