/* libiso8601/src/libiso8601/100_calc.c
 *
 *  (c)2006, Laurence Withers, <l@lwithers.me.uk>.
 *  Released under the GNU GPLv2. See file COPYING or
 *  http://www.gnu.org/copyleft/gpl.html for details.
*/



/* CALCULATION ROUTINES
 *
 *  This file contains calculation routines used internally in the library. Our date format is the
 *  number of days elapsed since 0000-001 (so that date would be 0, -0001-365 would be -1, etc.).
 *  Time is represented as the number of seconds elapsed since midnight at the start of the day. It
 *  is a value between 0 and 86399 (or 86400 for leap seconds).
 */

#define DAYS_IN_400_YEARS (146097)



int iso8601_isleap(int year)
{
        if(year % 4) return 0;
        if(year % 100) return 1;
        if(!(year % 400)) return 1;
        return 0;
}



/* struct monthcount, _days_in_month_common[], _days_in_month_leap[]
 *
 *  Tables of the number of days in each month, and the number of days elapsed since the start of
 *  the year for each month.
 */
struct monthcount {
    int elapsed, days;
};

static const struct monthcount _days_in_month_common[] = {
    { 0, 31 },
    { 31, 28 },
    { 59, 31 },
    { 90, 30 },
    { 120, 31 },
    { 151, 30 },
    { 181, 31 },
    { 212, 31 },
    { 243, 30 },
    { 273, 31 },
    { 304, 30 },
    { 334, 31 }
};

static const struct monthcount _days_in_month_leap[] = {
    { 0, 31 },
    { 31, 29 },
    { 60, 31 },
    { 91, 30 },
    { 121, 31 },
    { 152, 30 },
    { 182, 31 },
    { 213, 31 },
    { 244, 30 },
    { 274, 31 },
    { 305, 30 },
    { 335, 31 }
};



static int _days_in_month(int year, int month)
{
    const struct monthcount* mc;

    if(month < 1 || month > 12) {
        errno = EDOM;
        return -1;
    }

    mc = iso8601_isleap(year) ? _days_in_month_leap : _days_in_month_common;
    return mc[month - 1].days;
}



static void _to_year(int* year, int* days_left, const struct iso8601_date* date)
{
    div_t qr;
    int ndays = date->day;

    // Each 400 years have 97 leap days, giving 365*400+97 = DAYS_IN_400_YEARS days in 400 years
    qr = div(ndays, DAYS_IN_400_YEARS);
    *year = qr.quot * 400;
    ndays = qr.rem;

    // ensure that we always end up with between 0 and 146096 days remaining
    if(ndays < 0) {
        ndays += DAYS_IN_400_YEARS;
        *year -= 400;
    }

    // we insert `fake' leap days for years 101, 201, 301
    if(ndays >= 36890) ++ndays;
    if(ndays >= 73415) ++ndays;
    if(ndays >= 109940) ++ndays;

    // each remaining 100 year block has 24 leap days, giving 365*100+24 = 36524 days
    qr = div(ndays, 36525);
    *year += qr.quot * 100;
    ndays = qr.rem;

    // each 4-year block has 1 leap day, giving 365*4 + 1 = 1461 days
    qr = div(ndays, 1461);
    *year += qr.quot * 4;
    ndays = qr.rem;

    // the first year of a 4-year block has 1 leap day, 366 days
    if(ndays >= 366) {
        --ndays; // pretend to have dealt with leap day

        // 365 days per remaining year
        qr = div(ndays, 365);
        *year += qr.quot;
        ndays = qr.rem;
    }

    *days_left = ndays;
}



void iso8601_to_cal(int* year, int* month, int* day, const struct iso8601_date* date)
{
    const struct monthcount* mc;
    int ndays;

    // perform year calulation
    _to_year(year, &ndays, date);

    // now we simply have number of days elapsed since day 001 in `year'.
    mc = iso8601_isleap(*year) ? _days_in_month_leap : _days_in_month_common;
    *month = 1;
    while(ndays >= mc->days) {
        *month += 1;
        ndays -= mc->days;
        ++mc;
    }
    *day = ndays + 1;
}



void iso8601_to_ord(int* year, int* oday, const struct iso8601_date* date)
{
    int ndays;

    // perform year calcutation
    _to_year(year, &ndays, date);

    // now we simply have number of days elapsed since day 001 in `year'.
    *oday = ndays + 1;
}



static int _weekday_of_year(int year)
{
    int w = 6;

    /* Algorithm notes:
     *  - 0 = sun, 1 = mon, ..., 6 = sat
     *  - 0000-001 is a Saturday, day 6
     *  - every year we pass gives us one additional day (364 is divisible by 7)
     *  - but of course every leap year we pass gives us a further day
     *  - so for every 400 years, we add 497 (400 common years, 97 leap years); 497 % 7 = 0
     */
    year %= 400;
    if(year < 0) year += 400; // end up with between 0-399 years left
    w += year; // excluding leap years, we increase by 1 day a year
    w += (year + 3) / 4; // there is one leap year for every four years
    w -= (year - 1) / 100; // but one less for every century over 0
    w %= 7;

    return w;
}



void iso8601_to_week(int* year, int* week, int* wday, const struct iso8601_date* date)
{
    int ndays, w, has53 = 0;
    div_t d;

    // perform year and weekday calculation
    _to_year(year, &ndays, date);
    w = _weekday_of_year(*year);

    // find out what day jan 1 was; from there, we can find the ISO week and year number
    switch(w) {
    case 4: // W01 starts XXXY-12-28
        w += 7;
        has53 = 1; // years starting Thursday have 53 weeks
        break;

    case 5: // W01 starts XXXZ-01-03
    case 6: // W01 starts XXXZ-01-02
        break;

    case 0: // W01 starts XXXZ-01-01
    case 1: // W01 starts XXXY-12-31
    case 2: // W01 starts XXXY-12-30
        w += 7; // for week calculation
        break;

    case 3: // W01 starts XXXY-12-29
        w += 7;
        if(iso8601_isleap(*year)) has53 = 1; // leap years starting Wednesday have 53 weeks
    }

    // now we simply add the number of days elapsed since the start of the year, and % 7
    w += ndays;
    d = div(w, 7); // w can never be 0

    // do Sunday correction
    if(!d.rem) {
        d.rem = 7;
        --d.quot;
    }

    *wday = d.rem;
    if(d.quot) {
        if(d.quot == 53 && !has53) {
            d.quot = 1;
            *year += 1;
        }
        *week = d.quot;
    } else {
        *year -= 1;
        switch(_weekday_of_year(*year)) {
        case 3:
            *week = iso8601_isleap(*year) ? 53 : 52;
            break;

        case 4:
            *week = 53;
            break;

        default:
            *week = 52;
            break;
        }
    }
}



int _from_year(struct iso8601_date* date, int year)
{
    div_t qr;

    // check for range errors
    if(year < -5879609 || year > 5879609) {
        errno = ERANGE;
        return -1;
    }

    // Each 400 years have 97 leap days, giving 365*400+97 = DAYS_IN_400_YEARS days in 400 years
    qr = div(year, 400);
    date->day = qr.quot * DAYS_IN_400_YEARS;
    year = qr.rem;

    // ensure we have between 0 and 399 years
    if(year < 0) {
        date->day -= DAYS_IN_400_YEARS;
        year += 400;
    }

    // excluding leap days, there are 365 days per year
    date->day += 365 * year;
    date->day += (year + 3) / 4; // there is one leap year for every four years
    date->day -= (year - 1) / 100; // but one less for every century over 0
    return 0;
}



int iso8601_from_cal(struct iso8601_date* date, int year, int month, int day)
{
    const struct monthcount* mc;

    // check for domain errors
    mc = iso8601_isleap(year) ? _days_in_month_leap : _days_in_month_common;
    if(month < 1 || month > 12 || day < 1 || day > mc[month - 1].days) {
        errno = EDOM;
        return -1;
    }

    // perform year calculation
    if(_from_year(date, year)) return -1;

    // now get number of days elapsed up to start of month
    date->day += mc[month - 1].elapsed;

    // and add number of days elapsed sinced start of month
    date->day += day - 1;

    return 0;
}



int iso8601_from_ord(struct iso8601_date* date, int year, int oday)
{
    // check for domain errors
    if(oday < 1 || oday > (iso8601_isleap(year) ? 366 : 365)) {
        errno = EDOM;
        return -1;
    }

    // perform year calculation
    if(_from_year(date, year)) return -1;

    // now simply add number of days elapsed this year
    date->day += oday - 1;

    return 0;
}



int iso8601_from_week(struct iso8601_date* date, int isoyear, int week, int wday)
{
    int day1off, maxwk;

    // compute year part
    _from_year(date, isoyear);

    // 400-year cycle; ensure we're between 0-400 years
    isoyear %= 400;
    if(isoyear < 0) isoyear += 400;

    // domain check
    maxwk = (((isoyear % 7) == 52) || ((isoyear % 28) == 24)) ? 53 : 52;
    if(wday < 1 || wday > 7 || week < 1 || week > maxwk) {
        errno = EDOM;
        return -1;
    }

    /* Algorithm notes:
     *  We now compute the offset between the start day of the ISO year and the start day of the 
     *  real year. Year 0000 starts on a Saturday, meaning the ISO year 0000 starts on 0000-003
     *  (offset of +2 days). Each year reduces the offset by one day, and each leap year reduces it
     *  by a further day; the offset wraps from -3 to +3.
     */
    day1off = 2 - isoyear; // reduce offset by 1 for each year
    day1off += (isoyear - 1) / 100; // cancel out 1 leap year for each century past the first
    day1off -= (isoyear + 3) / 4; // 1 leap year every 4 years
    day1off %= 7;
    if(day1off < -3) day1off += 7;

    // now simply add in the day offset and days/weeks elapsed
    date->day += day1off + (week - 1) * 7 + wday - 1;

    return 0;
}



void iso8601_to_clocktime(int* hour, int* min, int* sec, const struct iso8601_date* date)
{
    div_t qr;

    // special case: leap second
    if(date->sec == 86400) {
        *hour = 23;
        *min = 59;
        *sec = 60;
        return;
    }

    // normal case
    qr = div(date->sec, 3600);
    *hour = qr.quot;
    qr = div(qr.rem, 60);
    *min = qr.quot;
    *sec = qr.rem;
}



int iso8601_from_clocktime(struct iso8601_date* date, int hour, int min, int sec)
{
    // special case: leap second
    if(hour == 23 && min == 59 && sec == 60) {
        date->sec = 86400;
        return 0;
    }

    // domain check
    if(hour < 0 || hour > 23 || min < 0 || min > 59 || sec < 0 || sec > 59) {
        errno = EDOM;
        return -1;
    }
    
    // normal calculation
    date->sec = hour * 3600 + min * 60 + sec;
    return 0;
}



/* options for text editors
kate: replace-trailing-space-save true; space-indent true; tab-width 4;
vim: expandtab:ts=4:sw=4
*/
