#!/bin/bash
# build.sh
#
#  Main build system entry point. Called with no arguments, it will rebuild every single package for
#  every single architecture. A build can be limited to a specific machine by setting BUILD_MACHINE.
#  A build can be limited to a specific package by setting BUILD_PACKAGE.
#
#  In the packages subdirectory, each package has its own build.sh file, which will set 
#  BUILD_PACKAGE and run this script. This script will call itself recursively to build specific
#  packages for specific machines.
#

#
#  Additional convenience behaviour:
#   If BUILD_PACKAGES_FROM is set, then we skip all packages in "list" before ${BUILD_PACKAGES_FROM}
#   occurs.
#
#   Unless FORCE_FRESH_BUILD is set, we only rebuild if the version number has changed. This is
#   implemented in the download functions.
#



# INITIAL PHASE ####################################################################################
#                                                                                                  #
#  If we are not performing a specific build (BUILD_MACHINE, BUILD_PACKAGE), recurse. If both      #
#  BUILD_MACHINE and BUILD_PACKAGE are set, fall through to the package builder.                   #
#                                                                                                  #
####################################################################################################



# automatically exit on failure
set -o pipefail
set -e

# read all configuration files
export LANG="C"
unset LC_CTYPE LC_COLLATE LC_ALL

if [ ! -e "build.version" ]
then
    cat <<EOF
Missing build.version . Create it with the following lines:
BUILD_LABEL="foo"    # e.g. lwithers@amethyst, platinum-stable, etc.
BUILD_VERSION="0"
EOF
    exit 1
fi

source "./build.version"
source "./conf.local"
source "./machine.conf"


# build tools if necessary
TOOL_DIR="$(cd tools && /bin/pwd)"
(
    cd "${TOOL_DIR}"
    make --silent all
)


# if BUILD_MACHINE is not set, recursively build for all machines
if [ -z "${BUILD_MACHINE}" ]
then
    for MACH in ${MACHINE_LIST}
    do
        BUILD_MACHINE="${MACH}" ./build.sh
    done

    exit 0
fi


# Select an input file checking for a machine specific version.
# Process it stripping blank lines and # comments and present the remainder to stdout.
# You get nothing if the file don't exist

select_and_cstrip() {
	local file="$1"

	[ -e "${file}-${BUILD_MACHINE}" ] && file="${file}-${BUILD_MACHINE}"
	
    [ -e "$file" ] && sed -e 's/#.*//' -e '/^[	 ]*$/d' $file
}


# if BUILD_PACKAGE is not set, recursively build all packages
if [ -z "${BUILD_PACKAGE}" ]
then
    am_building=0
    [ -z "${BUILD_PACKAGES_FROM}" ] && am_building=1

    while read -r PKG
    do
        [ "${PKG}" == "${BUILD_PACKAGES_FROM}" ] && am_building=1
        [ "${am_building}" -eq 0 ] && continue;
        BUILD_PACKAGE="${PKG}" ./build.sh
    done < <( select_and_cstrip "packages/list" )

    exit 0
fi

# the tree in which the filesystems live; used in build functions
DEST_FS_TREE="$(/bin/pwd)/filesystems/${BUILD_MACHINE}"



# BUILD FUNCTIONS ##################################################################################
#                                                                                                  #
#  A set of useful subroutines for building packages.                                              #
#                                                                                                  #
####################################################################################################



# build_check_latest()
#  Checks whether the version we have just downloaded/checked out matches the version we are about
#  to build. If so, and FORCE_FRESH_BUILD is not set, then we exit with status 0. Otherwise, we
#  store the version we are about to build in the version file in buildroot, and continue.
#
#  $1 -> version we are about to build
#
build_check_latest() {
    # store the version we are about to build in buildroot
    mkdir -p "${BUILDROOT}/usr/share/platinum-versions"
    echo "$1" > "${BUILDROOT}/usr/share/platinum-versions/${BUILD_PACKAGE}"

    # filename of latest build
    LATEST_BUILD_FILE="${DEST_FS_TREE}/usr/share/platinum-versions/${BUILD_PACKAGE}"

    echo "Checking for up-to-date:"
    echo "    Building -- $1"
    echo "    Latest   -- $(cat ${LATEST_BUILD_FILE})"

    # various tests
    if [ -n "${FORCE_FRESH_BUILD}" ]
    then
        echo "Forcing fresh build."
        return 0
    fi
    if [ auxbuild -nt "${LATEST_BUILD_FILE}" ]
    then
        echo "auxbuild updated, rebuilding."
        return 0
    fi
    if [ prune_list -nt "${LATEST_BUILD_FILE}" ]
    then
        echo "prune_list updated, rebuilding."
        return 0
    fi
    if [ -n "$(find patches -newer "${LATEST_BUILD_FILE}" 2>/dev/null)" ]
    then
        echo "patches updated, rebuilding."
        return 0
    fi
    if [ -n "$(find static -newer "${LATEST_BUILD_FILE}" 2>/dev/null)" ]
    then
        echo "static files updated, rebuilding."
        return 0
    fi

    # finally, check if we are about to build the same version; if so, abort
    if [ -e "${LATEST_BUILD_FILE}" ]
    then
        if [ "$(cat "${LATEST_BUILD_FILE}")" == "$1" ]
        then
            touch "Meta/up-to-date"
            echo "Already up to date."
            exit 0
        fi
    fi

    echo "Newer version available, rebuilding."
}



# build_git_clone()
#  Clones a git repository. If a remote branch is not specified, follows the origin's "master"
#  branch. If the repository is already cloned, simply fetches the latest changes. Then switches
#  to a specific tag or commit.
#
#  Note that if you change the remote branch option, you will need to destroy the local clone of
#  the origin so that the tracking branch is recreated.
#
#  $1 -> repository source URL
#  $2 -> tag/commit to switch to (e.g. "1.0.0" or "remotes/origin/master")
#  $3 -> destination package name
#  $4 -> remote branch
build_git_clone() {
    local ref

    if [ -d "Distn/$3/.git" ]
    then
        # update existing repository
        echo "Updating git repository \"$3\""
        ( cd "Distn/$3"
        git fetch origin
        git fetch --tags origin )
        [ $? -eq 0 ]
    else
        # check out new repository
        echo "Checking out git repository \"$3\" from \"$1\""
        mkdir -p "Distn"
        git clone "$1" "Distn/$3"

        # if a remote branch was specified, set up a tracking branch and switch to it
        if [ -n "$4" ]
        then
            ( cd "Distn/$3"
            git branch "platinum_builder_tracking_branch" "origin/$4"
            git checkout "platinum_builder_tracking_branch" )
            [ $? -eq 0 ]
        fi
    fi

    # switch to tag/commit
    if [ "$2" == "stable_or_development" ]
    then
        if grep -q "^${BUILD_PACKAGE}\$" "../development.local"
        then
            ref="remotes/origin/master"
        else
            ref="platinum-stable"
        fi
    else
        ref="$2"
    fi

    echo " - switching to \"${ref}\" in \"$1\""
    ( cd "Distn/$3"
    git reset --hard
    git checkout -q "${ref}" 
    rm -rf *
    git checkout-index --all --force )
    [ $? -eq 0 ]

    build_check_latest "$(cd "Distn/$3" && git rev-parse --verify HEAD)"
}



# build_download()
#  Downloads a file, storing it in Distn.
#   $1 -> URL
#   $2 -> destination file (default: from URL)
#   $3 -> set to a value to inhibit up-to-date checking
build_download() {
    if [ -z "$3" ]
    then
        build_check_latest "$1"
    fi

    DESTFILE="Distn/${2:-$(basename "$1")}"
    [ -s "${DESTFILE}" ] && return 0
    mkdir -p "Distn"

    if [ -f "/usr/portage/distfiles/$(basename "$1")" ] 
    then
        echo "Using portage distfile"
        cp "/usr/portage/distfiles/$(basename "$1")" "${DESTFILE}"
    elif ! wget -O "${DESTFILE}" -- "$1"
    then
        echo "Error downloading source"
        rm "${DESTFILE}"
        exit 1
    fi
}



# build_unpack()
#  Unpacks a tarball.
#   $1 -> tarball name
#   $2 -> directory to unpack in (default: ".")
build_unpack() {
    SRC="Distn/$1"
    DEST="Distn/${2:-.}"
    mkdir -p "${DEST}"

    case "${SRC}" in
    *.tar.bz2) UNPACKER="bunzip2" ;;
    *.tar.gz) UNPACKER="gunzip" ;;
    *.tar) UNPACKER="cat" ;;
    **)
        echo "Don't know how to unpack \"${SRC}\""
        exit 1
        ;;
    esac

    # HACK: remove directory of first entry in tarball
    X="$("${UNPACKER}" < "${SRC}" | tar -t -f - | head -n 1)" && true

    case "${X}" in
    /* | *..*)
        true 
        ;;
    *)
        rm -rf "${DEST}/${X//\/*}"
        ;;
    esac

#   This method of removing unwanted directories is probably better, but it
#   doesn't work if the tarball doesn't contain a directory entry for the
#   root directory.
#
#    for i in $("${UNPACKER}" < "${SRC}" | tar -t -f - | grep '^[^./][^/]*/$')
#    do
#        rm -rf "${DEST}/${i}"
#    done
    "${UNPACKER}" < "${SRC}" | tar -C "${DEST}" -x -f -
}



# build_patch()
#  Applies a patch or set of patches.
#   $1 -> directory to change to
#   $2 -> patch category (prefix)
#	$3 -> strip option (default --strip 1)
build_patch() {
	local strip="--strip 1"
	[ -n "$3" ] && strip="$3"

    echo "Applying patches in \"$1\"..."
    for PATCHFILE in patches/$2*
    do
        echo ${PATCHFILE}
        # catch bad globs
        [ ! -e "${PATCHFILE}" ] && continue

        echo " patching \"${PATCHFILE}\"..."
        patch --directory "Distn/$1" $strip < "${PATCHFILE}"
    done
}



# build_lw()
#  Builds a package using the LW build system.
#   $1 -> package name
#   $2 -> prefix (default /usr)
build_lw() {
    (
        cd "Distn/$1"
        export PREFIX="${2:-/usr}"
        export CC="${ARCH_CC_PREFIX}gcc"
        export CFLAGS="${ARCH_CFLAGS} ${ARCH_LDFLAGS}"
        export INSTALL_PREFIX="${BUILDROOT}"

        ./make.sh
        ./make.sh install
    )
    return $?
}



# build_autoconf()
#  Runs the autoconfigure script `./configure', with appropriate arguments.
#   $1 -> package name
#   $2 -> prefix (default /usr)
#   $3 -> additional arguments to configure
build_autoconf() {
    (
        cd "Distn/$1"
        PREFIX="${2:-/usr}"
        export CFLAGS="${ARCH_CFLAGS}"
        export CPPFLAGS="${ARCH_CPPFLAGS}"
        export LDFLAGS="${ARCH_LDFLAGS}"

		# If configure doesn't exist try to create it
		if [ ! -e configure ]
		then
			echo "run> autoconf"
			autoconf
		fi

        echo "run>" ./configure --prefix="${PREFIX}" --build="${ARCH_BUILD}" --host="${ARCH_HOST}" \
            --sysconfdir="/etc" --localstatedir="/var" --datadir="/usr/share" \
            --includedir="/usr/include" --mandir="/usr/share/man" \
            $3
        ./configure --prefix="${PREFIX}" --build="${ARCH_BUILD}" --host="${ARCH_HOST}" \
            --sysconfdir="/etc" --localstatedir="/var" --datadir="/usr/share" \
            --includedir="/usr/include" --mandir="/usr/share/man" \
            $3
    )
    return $?
}



# build_make()
#  Runs make in the specified package directory. Sets DESTDIR as well, in case this is an install.
#   $1    -> package name
#   $2..6 -> additional arguments to pass to make (config options, targets, ...)
# Care must be taken with arguments passed via multiple shell layers as some
# may contain spaces that must be preserved, yet argument may not be merged so
# simply using $* or "$*" doesn't hack it.
build_make() {
	local dir="Distn/$1"
	local p1="$2"
	local p2="$3"
	local p3="$4"
	local p4="$5"
	local p5="$6"

	export CFLAGS="${ARCH_CFLAGS}"
    export CPPFLAGS="${ARCH_CPPFLAGS}"
	export LDFLAGS="${ARCH_LDFLAGS}"
	export CC="${ARCH_CC_PREFIX}gcc"
	if [ -z "$p1" ]
	then
		make -C $dir DESTDIR="${BUILDROOT}"
	elif [ -z "$p2" ]
	then
		make -C $dir DESTDIR="${BUILDROOT}" "$p1"
	elif [ -z "$p3" ]
	then
		make -C $dir DESTDIR="${BUILDROOT}" "$p1" "$p2"
	elif [ -z "$p4" ]
	then
		make -C $dir DESTDIR="${BUILDROOT}" "$p1" "$p2" "$p3"
	elif [ -z "$p5" ]
	then
		make -C $dir DESTDIR="${BUILDROOT}" "$p1" "$p2" "$p3" "$p4"
	else
		make -C $dir DESTDIR="${BUILDROOT}" "$p1" "$p2" "$p3" "$p4" "$p5"
	fi
}


# strip_files()
# Strip file in the Build tree of unnecessary symbols.
strip_files() {
	find ${BUILDROOT} -type f -print | while read i
	do
		if file "$i" | grep -q "not stripped"
		then
			echo "Striping $i"
			chmod u+w "$i"
			${ARCH_CC_PREFIX}strip "$i"
		fi
	done
}



# PACKAGE BUILDER ##################################################################################
#                                                                                                  #
#  At this stage, we have BUILD_MACHINE and BUILD_PACKAGE, so we compare it against                #
#  the package's accepted machine list, and then we call the package-specific buildscript          #
#  (auxbuild).                                                                                     #
#                                                                                                  #
####################################################################################################



# set ARCH_HOST et al.
machine_setup "${BUILD_MACHINE}"

# test if this package can be built for this machine
for TEST_MACHINE in $(cat "packages/${BUILD_PACKAGE}/rules.machine")
do
    # "!*" inhibits all remaining machines
    [ "${TEST_MACHINE}" == "!*" ] && exit 0
    # "!MACHINE" inhibits a specific machine
    [ "${TEST_MACHINE}" == "!${BUILD_MACHINE}" ] && exit 0
    # "MACHINE" explicitly builds for a specific machine
    [ "${TEST_MACHINE}" == "${BUILD_MACHINE}" ] && break
done

echo "Building package \"${BUILD_PACKAGE}\" for machine \"${BUILD_MACHINE}\"..."

# change to package directory
cd "packages/${BUILD_PACKAGE}"

# ensure Meta exists
[ ! -d "Meta" ] && mkdir "Meta"

# ensure Buildroot exists and is empty
[ -e "Buildroot/${BUILD_MACHINE}" ] && rm -rf "Buildroot/${BUILD_MACHINE}"
mkdir -p "Buildroot/${BUILD_MACHINE}"
BUILDROOT="$(cd "Buildroot/${BUILD_MACHINE}" && /bin/pwd)"

# ensure crosslib (for libraries, header files and config scripts) exists
mkdir -p "../../crosslib/${BUILD_MACHINE}"
CROSSLIB="$(cd ../../crosslib/${BUILD_MACHINE} && /bin/pwd)"
mkdir -p "${CROSSLIB}"{/lib,/bin,/include}
ARCH_CFLAGS="${ARCH_CFLAGS} -I${CROSSLIB}/include"
ARCH_CPPFLAGS="-I${CROSSLIB}/include"
ARCH_LDFLAGS="${ARCH_LDFLAGS} -L${CROSSLIB}/lib"

# execute build script for the package
rm -f "Meta/up-to-date"
(
export PATH="${CROSSLIB}/bin:${PATH}"
export ARCH_CFLAGS="$ARCH_CFLAGS"
export ARCH_CPPFLAGS="$ARCH_CPPFLAGS"
export ARCH_LDFLAGS="$ARCH_LDFLAGS"
export ARCH_CC_PREFIX="$ARCH_CC_PREFIX"

# Autoconf and friends are dumb.  Even if a pkg-config occurs earlier in the
# $PATH they will use the one in /usr/bin/pkg-config unless you explicitly
# override.
export PKG_CONFIG_PATH=""
export PKG_CONFIG=$(which pkg-config)

export LIBEVENT_CFLAGS="-I${CROSSLIB}/include"
export LIBEVENT_LIBS="-L${CROSSLIB}/lib -levent"
export LIBGNUTLS_CFLAGS="-I${CROSSLIB}/include"
export LIBGNUTLS_LIBS="-L${CROSSLIB}/lib -lgnutls -lgnutls-extra -llzo2 -lgcrypt -lgpg-error -lz"
export LIBMODBUS_CFLAGS="-I${CROSSLIB}/include"
export LIBMODBUS_LIBS="-L${CROSSLIB}/lib -lmodbus"

if [ -e "${HOME}/.platinum-builder/redirect-stderr-to-Log" ]
then
    source ./auxbuild >& Log
else
    source ./auxbuild > Log
fi
)

if [ $? -ne 0 ]
then
    echo " * Build failed."
    exit 1
else
    if [ -e "Meta/up-to-date" ]
    then
        echo " * Already up to date."
        exit 0
    fi

    echo " * Build completed"
fi

# Now prune file from the prune_list if it exists (This is a per package pruning before we
# get to the main file system tidy)
while read prune
do
	for i in Buildroot/${BUILD_MACHINE}/$prune
	do
        rm -rf "${i}"
	done
done < <( select_and_cstrip prune_list )

# on multilib machines, possibly move "/lib" to "/lib64"
if [ -n "${MULTILIB_USE_64}" ]
then
    for t in lib usr/lib usr/local/lib
    do
        libdir="${BUILDROOT}/${t}"
        lib64dir="${BUILDROOT}/${t}64"
        if [ -e "${libdir}" ]
        then
            if [ -e "${lib64dir}" ]
            then
                echo " *** both \"${libdir}\" and \"${lib64dir}\" exist ***"
                exit 1
            fi
            mv "${libdir}" "${lib64dir}"
        fi
    done
fi



# FILESYSTEM MERGER ################################################################################
#                                                                                                  #
#  Once the package has been built, it must be merged into the filesystem. This requires post-     #
#  build processing, to remove any unwanted files and to save anything needed for building other   #
#  packages (localbuild). We also perform a pre-merge check to ensure that no files from other     #
#  packages are being overwritten. Finally, only changed files are copied, to avoid breaking rsync #
#  timestamps.                                                                                     #
#                                                                                                  #
####################################################################################################



if [ ! -d "${DEST_FS_TREE}" ]
then
    mkdir "${DEST_FS_TREE}"
fi

# on multilib machines, ensure lib and lib64 exist
if [ -n "${MULTILIB_USE_64}" ]
then
    if [ ! -e "${DEST_FS_TREE}/lib" ]
    then
        mkdir -p "${DEST_FS_TREE}/lib64"
        ln -sf "lib64" "${DEST_FS_TREE}/lib"
    fi

    if [ ! -e "${DEST_FS_TREE}/usr/lib" ]
    then
        mkdir -p "${DEST_FS_TREE}/usr/lib64"
        ln -sf "lib64" "${DEST_FS_TREE}/usr/lib"
    fi

    if [ ! -e "${DEST_FS_TREE}/usr/local/lib" ]
    then
        mkdir -p "${DEST_FS_TREE}/usr/local/lib64"
        ln -sf "lib64" "${DEST_FS_TREE}/usr/local/lib"
    fi
fi



# Function to build symlinks from a shared object file. First argument is path to shared object.
# Second argument is "development" to build a development symlink (raw .so) or "soname" to build
# a soname symlink.
build_shared_object_symlink() {
    (
        cd "$(dirname "$1")"

        # get library's soname
        SONAME="$(readelf -d "$(basename "$1")" | grep 'Library soname' | sed -e 's,^.*\[\(.*\)\].*$,\1,')"
        [ -z "${SONAME}" ] && exit 0

        # build symlink name
        case "$2" in
        development)
            LINKNAME="${SONAME/.so*/.so}"
            case "${LINKNAME}" in
            libc.so)
                exit 0 # don't override system's glibc
                ;;
            libevent-*.so | libgobj-*.so | libusb-*.so)
                # these libs put version number before .so in soname
                LINKNAME="$(echo "${LINKNAME}" | sed -e 's,-[^-]*\.so,.so,')"
                ;;
            esac
            ;;
        soname)
            LINKNAME="${SONAME}"
            ;;
        esac
        [ "${SONAME}" == "${LINKNAME}" ] && exit 0

        # make symlink
        rm -f "${LINKNAME}"
        echo "Linking $(basename "$1") to ${LINKNAME}"
        ln -s "$(basename "$1")" "${LINKNAME}"
    )
}

# build list of files in Buildroot
TMP_FILELIST="$(mktemp)"
( cd "${BUILDROOT}" && find . > "${TMP_FILELIST}" )

# process each file in Buildroot
while read -r FILE
do
    # files which are not needed for either crosslib or final image are removed
    case "${FILE}" in
    */lib/*.la | */lib/*.a | */lib64/*.la | */lib64/*.a )
        rm -f "${BUILDROOT}/${FILE}"
        continue
        ;;
    esac

    # files which live under Python's site-packages dir are ignored by crosslib
    case "${FILE}" in
    *lib/python*/site-packages*)
        continue
        ;;
    esac

    # copy anything from include or lib directories, or from bin if it's a lib-config script, into a
    # crosslib directory (used for later builds to link against)
    PARENT_DIR="$(basename "$(dirname "${FILE}")")"
    case "${PARENT_DIR}" in
    lib | lib64)
        case "$(file "${BUILDROOT}/${FILE}")" in
        *"shared object"*)
            # copy across file
            cp -af "${BUILDROOT}/${FILE}" "${CROSSLIB}/lib"
            build_shared_object_symlink "${CROSSLIB}/lib/$(basename "${FILE}")" "development"
            ;;
        esac
        ;;

    include)
        cp -a "${BUILDROOT}/${FILE}" "${CROSSLIB}/${PARENT_DIR}"
        ;;

    bin)
        if [ -z "${FILE/*-config}" ]
        then
            sed -e "s,^prefix=.*$,prefix=\"${CROSSLIB}\"," \
                -e "s,^include_dir=.*$,include_dir=\"${CROSSLIB}/include\"," \
                -e "s,^includedir=.*$,includedir=\"${CROSSLIB}/include\"," \
                -e "s,^lib_dir=.*$,lib_dir=\"${CROSSLIB}/lib\"," \
                -e "s,^libdir=.*$,libdir=\"${CROSSLIB}/lib\"," \
                < "${BUILDROOT}/${FILE}" \
                > "${CROSSLIB}/${PARENT_DIR}/$(basename "${FILE}")"
            chmod 0755 "${CROSSLIB}/${PARENT_DIR}/$(basename "${FILE}")"
            rm "${BUILDROOT}/${FILE}"
        fi
        ;;
    esac

    # remove library symlinks
    if [ -h "${BUILDROOT}/${FILE}" ]
    then
        case "$(basename "${FILE}")" in
        lib*.so | lib*.so.*)
            rm "${BUILDROOT}/${FILE}"
            ;;
        esac
    fi
done < "${TMP_FILELIST}"

# remove other files we don't want to merge into the filesystem
rm -rf \
    "${BUILDROOT}/include" \
    "${BUILDROOT}/usr/include" \
    "${BUILDROOT}/usr/share/locale" \
    "${BUILDROOT}/usr/doc" \
    "${BUILDROOT}/usr/share/doc" \
    "${BUILDROOT}/lib/pkgconfig" \
    "${BUILDROOT}/lib64/pkgconfig" \
    "${BUILDROOT}/usr/lib/pkgconfig" \
    "${BUILDROOT}/usr/lib64/pkgconfig" \
    "${BUILDROOT}/usr/share/man" \
    "${BUILDROOT}/man" \
    "${BUILDROOT}/usr/man" \
    "${BUILDROOT}/share/man" \
    "${BUILDROOT}/info" \
    "${BUILDROOT}/usr/info" \
    "${BUILDROOT}/usr/share/info"

# build pruned file list
( cd "${BUILDROOT}" && find . > "${TMP_FILELIST}" )

# if this is the first time building the package, create empty filelist
FILELIST="${CROSSLIB}/share/filelist/${BUILD_PACKAGE}"
if [ ! -e "${FILELIST}" ]
then
    mkdir -p "$(dirname "${FILELIST}")"
    touch "${FILELIST}"
fi

# check if any files conflict with other packages
while read -r FILE
do
    [ -d "${BUILDROOT}/${FILE}" -a -d "${DEST_FS_TREE}/${FILE}" ] && continue
    [ ! -e "${DEST_FS_TREE}/${FILE}" ] && continue
	# Escape [ in filenames
	EFILE=$(echo "${FILE}" | sed -e 's/\\/\\\\/g' -e 's/\[/\\[/g')
    if ! grep -q "^${EFILE}\$" "${FILELIST}"
    then
        echo "*** ERROR: overwriting existing file \"${FILE}\" from another package."
        exit 1
    fi

    if [ -n "${MULTILIB_USE_64}" ]
    then
        case "${FILE}" in
        ./lib/*|./usr/lib/*|./usr/local/lib/*)
            echo "Cannot merge with file in /lib: \"${FILE}\""
            exit 1
            ;;
        esac
    fi
done < "${TMP_FILELIST}"

# actually merge in files
SO_CHANGED="0"
while read -r FILE
do
    if [ "${FILE}" == "." ]; then continue; fi
    SRC="${BUILDROOT}/${FILE}"
    DEST="${DEST_FS_TREE}/${FILE}"

    "${TOOL_DIR}/Pt-merge" "${DEST}" "${SRC}" "${FILE}"

    # set SO_CHANGED if this is a library
    case "$(basename "$(dirname "${FILE}")")" in
    lib | lib64)
        case "$(file "${SRC}")" in
        *"shared object"*)
            SO_CHANGED="1"
            ;;
        esac
        ;;
    esac
done < "${TMP_FILELIST}"

# unmerge old files, update local list
TMP_COMMLIST="$(mktemp)"
TMP_FILELIST_SORTED="$(mktemp)"
sort --output "${TMP_FILELIST_SORTED}" "${TMP_FILELIST}"
mv "${TMP_FILELIST_SORTED}" "${TMP_FILELIST}"
comm -2 -3 "${FILELIST}" "${TMP_FILELIST}" | \
    sort --reverse --output "${TMP_COMMLIST}"

while read -r FILE
do
    if [ -d "${DEST_FS_TREE}/${FILE}" -a \! -h "${DEST_FS_TREE}/${FILE}" ]
    then
        rmdir --ignore-fail-on-non-empty "${DEST_FS_TREE}/${FILE}"
    else
        rm --force "${DEST_FS_TREE}/${FILE}"
    fi
done < "${TMP_COMMLIST}"

rm "${TMP_COMMLIST}"
mv "${TMP_FILELIST}" "${FILELIST}"
chmod 0644 "${FILELIST}"

# HACK: ensure that the version file is newer than auxbuild etc.
touch "${DEST_FS_TREE}/usr/share/platinum-versions/${BUILD_PACKAGE}"

# update build.version
(
    cd "../.."
    sed -e "s,^BUILD_VERSION=.*\$,BUILD_VERSION=\"$((BUILD_VERSION+1))\"," -i "build.version"
    for MACH in ${MACHINE_LIST}
    do
        mkdir -p "filesystems/${MACH}/etc"
        cp "build.version" "filesystems/${MACH}/etc/build.version"
    done
)
[ $? -eq 0 ]

# update shared libraries
if [ "${SO_CHANGED}" -ne 0 ]
then
    # update symlinks
    if [ -n "${MULTILIB_USE_64}" ]
    then
        /sbin/ldconfig -n "${DEST_FS_TREE}/lib64" "${DEST_FS_TREE}/usr/lib64"
    else
        /sbin/ldconfig -n "${DEST_FS_TREE}/lib" "${DEST_FS_TREE}/usr/lib"
    fi

    # reverse dependency check
    (
        export BUILD_MACHINE
        cd "../.."
        ./check-revdep.sh
    )
    [ $? -eq 0 ]
fi

# exit cleanly
echo " * build finished successfully"
exit 0

# vim: ts=4:sw=4:expandtab
