#!/bin/bash
#  vim: ts=4:sw=4:expandtab
#
#  This script builds a set of files to fulfill our open source license
#  requirement. This set of files is to be rsynced to a web server allowing
#  general public access.
#
#  For each package that has a license marking it requiring source
#  distribution, we copy over the build scripts and a pointer to the source
#  code. The repository also has a copy of all downloaded tarballs. git
#  repositories are handled separately.
#

[ ! -e "opensource" ] && exit 0

set -e
source "./build.version"
source "./machine.conf"

TARBALLDIR="opensource/tarballs"

# new source rsync directory
RSYNCDIR="opensource/${BUILD_LABEL}-${BUILD_VERSION}"
mkdir "${RSYNCDIR}"
mkdir "${RSYNCDIR}/files"

# create directory in which we will assemble our open source files
DESTDIR="${RSYNCDIR}/platinum-builder"
mkdir "${DESTDIR}"
mkdir "${DESTDIR}/packages"

# copy over static parts of build system that are required
cp *.sh build.version machine.conf "${DESTDIR}"
mkdir "${DESTDIR}/tools"
cp tools/Makefile tools/pkg-config tools/*.c "${DESTDIR}/tools"

# begin HTML index file for the package
cat > "${RSYNCDIR}/index.html" <<EOF
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN"
 "http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd">

<html xmlns="http://www.w3.org/1999/xhtml" lang="en" xml:lang="en">
<head>
<link rel='stylesheet' type='text/css' media='screen' href='/platinum/platinum.css' />
<title>Release ${BUILD_LABEL}-${BUILD_VERSION} - Open Source - Platinum - Guralp Systems Ltd.</title>
</head>

<body>
<p class='navbar'><a href='/'>G&uuml;ralp Systems Ltd.</a> |
<a href='/platinum/'>Platinum</a> |
<a href='/platinum/opensource/'>Opensource</a> |
Release ${BUILD_LABEL}-${BUILD_VERSION}</p>

<h1>Platinum: open-source: release ${BUILD_LABEL}-${BUILD_VERSION}</h1>
<p>A list of all open source packages may be found <a href='files/'>here</a>.
Individual packages are listed below. Git repositories may be checked out with
<code>git clone git://git.guralp.com/<em>repository</em></code> and the correct
revision with <code>git checkout <em>revision</em></code> in the checked-out
directory.</p>

<p>The build scripts for all packages may be downloaded
<a href='files/platinum-builder.tar.gz'>here</a>.</p>
EOF

# copy over required per-package data
while read -r BUILD_PACKAGE
do
    [ "${BUILD_PACKAGE:0:1}" == "#" ] && continue
    [ -z "${BUILD_PACKAGE}" ] && continue

    # skip packages without a license
    LICENSE_FILE="packages/${BUILD_PACKAGE}/license"
    [ -e "${LICENSE_FILE}" ] || continue
    LICENSE="$(cat "${LICENSE_FILE}")"

    # test whether we need to include this or not
#    COPY_SOURCE="0"
#    case "${LICENSE}" in
#    *GPL* | *BSD* | gnuplot | MIT | openssh | openssl | postgresql)
#        COPY_SOURCE="1"
#        ;;
#    esac
#    [ "${COPY_SOURCE}" -eq 0 ] && continue

    # create a copy of the build files
    echo "${BUILD_PACKAGE}" >> "${DESTDIR}/packages/list"
    mkdir "${DESTDIR}/packages/${BUILD_PACKAGE}"
    for PART in auxbuild patches static build.sh rules.machine
    do
        [ -e "packages/${BUILD_PACKAGE}/${PART}" ] || continue
        cp -a "packages/${BUILD_PACKAGE}/${PART}" "${DESTDIR}/packages/${BUILD_PACKAGE}"
    done

    cat >> "${RSYNCDIR}/index.html" <<EOF

<h2>${BUILD_PACKAGE}</h2>
<p><strong>License</strong>: ${LICENSE}</p>
<ul>
EOF

    # now do some trickery to work out which source files are required
    SOURCE_PARTS=""
    for BUILD_MACHINE in ${MACHINE_LIST}
    do
        SOURCE_PARTS_FILE="filesystems/${BUILD_MACHINE}/usr/share/platinum-versions/${BUILD_PACKAGE}"
        [ -e "${SOURCE_PARTS_FILE}" ] || continue
        for SOURCE_PART in $(cat "${SOURCE_PARTS_FILE}")
        do
            case "${SOURCE_PARTS}" in
            *"${SOURCE_PART}"*)
                ;;
            *)
                SOURCE_PARTS="${SOURCE_PARTS} ${SOURCE_PART}"
                ;;
            esac
        done
    done

    # store pointers to source files
    for SOURCE_PART in ${SOURCE_PARTS}
    do
        if echo "${SOURCE_PART}" | grep -qE "^[0-9a-f]{40,40}$"
        then
            # git repository
            echo "git://git.guralp.com/${BUILD_PACKAGE} ${SOURCE_PART}" >> "${DESTDIR}/packages/${BUILD_PACKAGE}/source_files"
            cat >> "${RSYNCDIR}/index.html" <<EOF
<li>
Git repository <code>git://git.guralp.com/${BUILD_PACKAGE}</code> revision <code>${SOURCE_PART}</code>.
</li>
EOF
        else
            # source tarball
            SOURCE_FILE="$(basename "${SOURCE_PART}")"
            echo "${SOURCE_PART}" >> "${DESTDIR}/packages/${BUILD_PACKAGE}/source_files"

            cat >> "${RSYNCDIR}/index.html" <<EOF
<li>
<a href='files/${SOURCE_FILE}'>${SOURCE_FILE}</a>
</li>
EOF

            ln -sf "../../tarballs/${SOURCE_FILE}" "${RSYNCDIR}/files/${SOURCE_FILE}"
            if [ ! -e "${TARBALLDIR}/${SOURCE_FILE}" ]
            then
                cp "packages/${BUILD_PACKAGE}/Distn/${SOURCE_FILE}" "${TARBALLDIR}/${SOURCE_FILE}"
            fi
        fi
    done

    echo "</ul>" >> "${RSYNCDIR}/index.html"
done < "packages/list"

# build tarball
tar -C "${RSYNCDIR}" -c "platinum-builder" | gzip --best > "${RSYNCDIR}/files/platinum-builder.tar.gz"

# clean up
rm -rf "${DESTDIR}"
cat >> "${RSYNCDIR}/index.html" <<EOF

</body>
</html>
EOF

# insert top-level link
echo -n "" > "opensource/index.html.new"
while read -r LINE
do
    echo "${LINE}" >> "opensource/index.html.new"
    case "${LINE}" in
    *"<?platinum-builder insert-toplink?>"*)
        cat >> "opensource/index.html.new" <<EOF
<li>
<a href='${BUILD_LABEL}-${BUILD_VERSION}/'>${BUILD_LABEL}-${BUILD_VERSION} ($(isodate -es))</a>
</li>

EOF
        ;;
    esac
done < "opensource/index.html"
mv "opensource/index.html.new" "opensource/index.html"
